<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

require_once __DIR__ . '/../ai/_gemini.php';
require_once __DIR__ . '/../ai/_limits.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
require_admin($me);

$body = read_json();
require_fields($body, ['bankId']);

$bankId = trim((string)$body['bankId']);
$limit = (int)($body['limit'] ?? 20);
if ($limit < 1) $limit = 1;
if ($limit > 50) $limit = 50;

// Fetch bank/course metadata
$b = $pdo->prepare('SELECT b.id, b.title, b.course_id AS courseId, c.code, c.title AS courseTitle
                    FROM banks b
                    LEFT JOIN courses c ON c.id = b.course_id
                    WHERE b.id = :id LIMIT 1');
$b->execute([':id' => $bankId]);
$bank = $b->fetch();
if (!$bank) json_response(['success' => false, 'error' => 'Bank not found'], 404);

// Find missing explanations
$lim = (int)$limit;
$q = $pdo->query("SELECT id, prompt, `options`, answer_index AS answerIndex
                  FROM questions
                  WHERE bank_id = " . $pdo->quote($bankId) . " AND (explanation IS NULL OR TRIM(explanation) = '')
                  ORDER BY sort_order ASC
                  LIMIT {$lim}");
$rows = $q->fetchAll() ?: [];

if (count($rows) === 0) {
  json_response(['success' => true, 'data' => ['updated' => 0, 'message' => 'No missing explanations found']]);
}

// Rate limit + daily quota
ai_enforce_limits($pdo, (string)$me['id'], 'fill_explanations');
$callHash = hash('sha256', json_encode(['bankId' => $bankId, 'count' => count($rows)], JSON_UNESCAPED_SLASHES));
ai_log_call($pdo, (string)$me['id'], 'fill_explanations', $callHash);

$courseCode = (string)($bank['code'] ?? $bank['courseId']);
$courseTitle = (string)($bank['courseTitle'] ?? '');
$bankTitle = (string)($bank['title'] ?? '');

$payload = [];
foreach ($rows as $r) {
  $payload[] = [
    'id' => (string)$r['id'],
    'prompt' => (string)$r['prompt'],
    'options' => json_decode((string)$r['options'], true),
    'answerIndex' => (int)$r['answerIndex'],
  ];
}

$prompt = "You are a calm, accurate tutor for Nigerian university students.\n"
  . "Safety rules: do not include hate/harassment/sexual content, do not provide illegal instructions, and do not request personal data.\n"
  . "Task: write short explanations for multiple-choice questions.\n"
  . "Course: {$courseCode}" . ($courseTitle !== '' ? " ({$courseTitle})" : '') . "\n"
  . "Bank: {$bankTitle}\n\n"
  . "Return ONLY valid JSON (no markdown, no extra text) in this exact shape:\n"
  . "{\"items\":[{\"id\":\"...\",\"explanation\":\"...\"}]}\n\n"
  . "Rules:\n"
  . "- Explanations must be 1-3 sentences, clear and simple.\n"
  . "- Do NOT repeat all options; focus on why the correct option is correct (and briefly why others are wrong if needed).\n"
  . "- Do not add new questions.\n\n"
  . "QUESTIONS JSON:\n" . json_encode(['questions' => $payload], JSON_UNESCAPED_SLASHES);

$resp = gemini_generate_text($prompt, [
  'temperature' => 0.3,
  'maxOutputTokens' => 1400,
]);

if (!$resp['ok']) {
  json_response(['success' => false, 'error' => $resp['error'] ?? 'AI failed'], (int)($resp['status'] ?? 502));
}

$text = trim((string)$resp['text']);
$jsonText = ai_extract_json($text);
$data = $jsonText !== null ? json_decode($jsonText, true) : null;

if (!is_array($data) || !isset($data['items']) || !is_array($data['items'])) {
  json_response(['success' => false, 'error' => 'AI returned invalid format'], 502);
}

$items = $data['items'];
$map = [];
foreach ($items as $it) {
  if (!is_array($it)) continue;
  $id = trim((string)($it['id'] ?? ''));
  $ex = trim((string)($it['explanation'] ?? ''));
  if ($id !== '' && $ex !== '') $map[$id] = $ex;
}

if (count($map) === 0) {
  json_response(['success' => false, 'error' => 'AI returned empty explanations'], 502);
}

$upd = $pdo->prepare('UPDATE questions SET explanation = :e WHERE bank_id = :bid AND id = :id');

$updated = 0;
foreach ($rows as $r) {
  $qid = (string)$r['id'];
  if (!isset($map[$qid])) continue;
  $ex = $map[$qid];

  // hard cap explanation length
  if (mb_strlen($ex) > 900) $ex = mb_substr($ex, 0, 900);

  $upd->execute([':e' => $ex, ':bid' => $bankId, ':id' => $qid]);
  $updated++;
}

json_response([
  'success' => true,
  'data' => [
    'updated' => $updated,
    'totalRequested' => count($rows),
    'model' => gemini_model(),
  ]
]);
