<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

require_once __DIR__ . '/../ai/_gemini.php';
require_once __DIR__ . '/../ai/_limits.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
require_admin($me);

$body = read_json();
require_fields($body, ['bankId']);

$bankId = trim((string)$body['bankId']);
$limit = (int)($body['limit'] ?? 20);
if ($limit < 1) $limit = 1;
if ($limit > 50) $limit = 50;

// Fetch bank/course metadata
$b = $pdo->prepare('SELECT b.id, b.title, b.course_id AS courseId, b.mode, c.code, c.title AS courseTitle
                    FROM banks b
                    LEFT JOIN courses c ON c.id = b.course_id
                    WHERE b.id = :id LIMIT 1');
$b->execute([':id' => $bankId]);
$bank = $b->fetch();
if (!$bank) json_response(['success' => false, 'error' => 'Bank not found'], 404);

if (strtolower((string)($bank['mode'] ?? '')) !== 'theory') {
  json_response(['success' => false, 'error' => 'This bank is not a theory bank'], 422);
}

$lim = (int)$limit;

// Select questions missing guide OR points
$q = $pdo->query(
  "SELECT id, prompt, guide, points_json AS pointsJson
   FROM theory_questions
   WHERE bank_id = " . $pdo->quote($bankId) . "
     AND (
       guide IS NULL OR TRIM(guide) = ''

       OR points_json IS NULL OR TRIM(points_json) = '' OR points_json = '[]'
     )
   ORDER BY sort_order ASC, id ASC
   LIMIT {$lim}"
);
$rows = $q->fetchAll() ?: [];

if (count($rows) === 0) {
  json_response(['success' => true, 'data' => ['updated' => 0, 'message' => 'No theory questions missing guide/points were found.']]);
}

// Rate limit + daily quota
ai_enforce_limits($pdo, (string)$me['id'], 'fill_theory_guides');
$callHash = hash('sha256', json_encode(['bankId' => $bankId, 'count' => count($rows)], JSON_UNESCAPED_SLASHES));
ai_log_call($pdo, (string)$me['id'], 'fill_theory_guides', $callHash);

$courseCode = (string)($bank['code'] ?? $bank['courseId']);
$courseTitle = (string)($bank['courseTitle'] ?? '');
$bankTitle = (string)($bank['title'] ?? '');

$payload = [];
foreach ($rows as $r) {
  $payload[] = [
    'id' => (string)($r['id'] ?? ''),
    'prompt' => (string)($r['prompt'] ?? ''),
    'hasGuide' => ($r['guide'] !== null && trim((string)$r['guide']) !== ''),
    'hasPoints' => ($r['pointsJson'] !== null && trim((string)$r['pointsJson']) !== ''),
  ];
}

$prompt = "You are a careful exam tutor and marking-scheme writer for Nigerian university students.\n"
  . "Safety rules: do not include hate/harassment/sexual content, do not provide illegal instructions, and do not request personal data.\n"
  . "Task: Generate marking guides for theory questions.\n"
  . "Course: {$courseCode}" . ($courseTitle !== '' ? " ({$courseTitle})" : '') . "\n"
  . "Bank: {$bankTitle}\n\n"
  . "Return ONLY valid JSON (no markdown, no extra text) in this exact shape:\n"
  . "{\"items\":[{\"id\":\"...\",\"guide\":\"...\",\"points\":[\"...\"]}]}\n\n"
  . "Rules:\n"
  . "- guide: concise model guide (max 8-12 lines).\n"
  . "- points: 5-10 marking points; each point short and specific.\n"
  . "- Do not include overly long essays.\n"
  . "- If the prompt is unclear, still try your best but keep points general.\n\n"
  . "QUESTIONS (some already have guide/points; still return both fields for every item):\n"
  . json_encode(['questions' => $payload], JSON_UNESCAPED_SLASHES);

$resp = gemini_generate_text($prompt, [
  'temperature' => 0.35,
  'maxOutputTokens' => 1800,
]);

if (!$resp['ok']) {
  json_response(['success' => false, 'error' => $resp['error'] ?? 'AI failed'], (int)($resp['status'] ?? 502));
}

$text = trim((string)$resp['text']);
$jsonText = ai_extract_json($text);
$data = $jsonText !== null ? json_decode($jsonText, true) : null;

if (!is_array($data) || !isset($data['items']) || !is_array($data['items'])) {
  json_response(['success' => false, 'error' => 'AI returned invalid format'], 502);
}

$map = [];
foreach ($data['items'] as $it) {
  if (!is_array($it)) continue;
  $id = trim((string)($it['id'] ?? ''));
  $guide = trim((string)($it['guide'] ?? ''));
  $pts = $it['points'] ?? [];
  $points = [];
  if (is_array($pts)) {
    foreach ($pts as $p) {
      $s = trim((string)$p);
      if ($s !== '') $points[] = $s;
      if (count($points) >= 12) break;
    }
  }

  if ($id !== '') {
    $map[$id] = [
      'guide' => $guide,
      'points' => $points,
    ];
  }
}

if (count($map) === 0) {
  json_response(['success' => false, 'error' => 'AI returned empty items'], 502);
}

$updGuide = $pdo->prepare('UPDATE theory_questions SET guide = :g WHERE bank_id = :bid AND id = :id');
$updPoints = $pdo->prepare('UPDATE theory_questions SET points_json = :p WHERE bank_id = :bid AND id = :id');

$updatedQuestions = 0;
$updatedGuides = 0;
$updatedPoints = 0;

foreach ($rows as $r) {
  $qid = (string)($r['id'] ?? '');
  if ($qid === '' || !isset($map[$qid])) continue;

  $existingGuide = $r['guide'] === null ? '' : trim((string)$r['guide']);
  $existingPtsRaw = $r['pointsJson'] === null ? '' : trim((string)$r['pointsJson']);

  $didAny = false;

  // Fill guide if missing
  if ($existingGuide === '') {
    $g = (string)($map[$qid]['guide'] ?? '');
    if ($g !== '') {
      if (mb_strlen($g) > 2400) $g = mb_substr($g, 0, 2400);
      $updGuide->execute([':g' => $g, ':bid' => $bankId, ':id' => $qid]);
      $updatedGuides++;
      $didAny = true;
    }
  }

  // Fill points if missing
  $needsPoints = true;
  if ($existingPtsRaw !== '') {
    $tmp = json_decode($existingPtsRaw, true);
    if (is_array($tmp) && count($tmp) > 0) $needsPoints = false;
  }

  if ($needsPoints) {
    $pts = $map[$qid]['points'] ?? [];
    if (is_array($pts) && count($pts) > 0) {
      $json = json_encode(array_values($pts), JSON_UNESCAPED_UNICODE);
      $updPoints->execute([':p' => $json, ':bid' => $bankId, ':id' => $qid]);
      $updatedPoints++;
      $didAny = true;
    }
  }

  if ($didAny) $updatedQuestions++;
}

json_response([
  'success' => true,
  'data' => [
    'updatedQuestions' => $updatedQuestions,
    'updatedGuides' => $updatedGuides,
    'updatedPoints' => $updatedPoints,
    'totalRequested' => count($rows),
    'model' => gemini_model(),
  ]
]);
