<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

require_once __DIR__ . '/../ai/_gemini.php';
require_once __DIR__ . '/../ai/_limits.php';

// POST /admin/ai/quality-audit
// Admin-only: quality audit a question bank (MCQ or Theory)
// - Always runs cheap heuristic checks
// - Optionally runs an AI audit on a limited sample

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
require_admin($me);

$body = read_json();
require_fields($body, ['bankId']);

$bankId = trim((string)($body['bankId'] ?? ''));
$limit = (int)($body['limit'] ?? 80);
if ($limit < 1) $limit = 1;
if ($limit > 200) $limit = 200;

$includeAi = (bool)($body['includeAi'] ?? false);
$aiLimit = (int)($body['aiLimit'] ?? 10);
if ($aiLimit < 1) $aiLimit = 1;
if ($aiLimit > 25) $aiLimit = 25;

if ($bankId === '') {
  json_response(['success' => false, 'error' => 'Missing bankId'], 422);
}

// Fetch bank metadata
$b = $pdo->prepare('SELECT b.id, b.title, b.course_id AS courseId, b.mode,
                           c.code AS courseCode, c.title AS courseTitle
                    FROM banks b
                    LEFT JOIN courses c ON c.id = b.course_id
                    WHERE b.id = :id
                    LIMIT 1');
$b->execute([':id' => $bankId]);
$bank = $b->fetch();
if (!$bank) json_response(['success' => false, 'error' => 'Bank not found'], 404);

$mode = strtolower((string)($bank['mode'] ?? 'mcq'));
$isTheory = $mode === 'theory';

// --- Helpers
function qa_norm(string $s): string {
  $s = trim($s);
  if ($s === '') return '';
  if (function_exists('mb_strtolower')) $s = (string)mb_strtolower($s, 'UTF-8');
  else $s = strtolower($s);
  // collapse whitespace + strip punctuation
  $s = preg_replace('/[^\p{L}\p{N}]+/u', ' ', $s) ?? $s;
  $s = preg_replace('/\s+/u', ' ', $s) ?? $s;
  return trim($s);
}

function qa_preview(string $s, int $n = 140): string {
  $s = trim($s);
  if ($s === '') return '';
  $s = preg_replace('/\s+/u', ' ', $s) ?? $s;
  $len = function_exists('mb_strlen') ? (int)mb_strlen($s, 'UTF-8') : strlen($s);
  if ($len <= $n) return $s;
  $cut = function_exists('mb_substr') ? (string)mb_substr($s, 0, $n, 'UTF-8') : substr($s, 0, $n);
  return rtrim($cut) . '…';
}

function qa_add_issue(array &$issues, string $qid, string $severity, string $code, string $message, string $promptPreview, array $extra = []): void {
  $issues[] = array_merge([
    'questionId' => $qid,
    'severity' => $severity, // low|medium|high
    'code' => $code,
    'message' => $message,
    'promptPreview' => $promptPreview,
  ], $extra);
}

// --- Load questions (limited)
$rows = [];
try {
  if ($isTheory) {
    $sql = 'SELECT id, prompt, guide, points_json AS pointsJson
            FROM theory_questions
            WHERE bank_id = :bid
            ORDER BY sort_order ASC, id ASC
            LIMIT ' . (int)$limit;
  } else {
    $sql = 'SELECT id, prompt, options, answer_index AS answerIndex, explanation
            FROM questions
            WHERE bank_id = :bid
            ORDER BY sort_order ASC, id ASC
            LIMIT ' . (int)$limit;
  }
  $st = $pdo->prepare($sql);
  $st->execute([':bid' => $bankId]);
  $rows = $st->fetchAll() ?: [];
} catch (Throwable $e) {
  json_response(['success' => false, 'error' => 'Failed to load questions', 'detail' => $e->getMessage()], 500);
}

if (count($rows) === 0) {
  json_response([
    'success' => true,
    'data' => [
      'bank' => [
        'id' => (string)$bank['id'],
        'title' => (string)($bank['title'] ?? ''),
        'courseId' => (string)($bank['courseId'] ?? ''),
        'mode' => $mode,
      ],
      'summary' => [
        'totalQuestions' => 0,
        'heuristicIssueCount' => 0,
        'questionsWithIssues' => 0,
      ],
      'issues' => [],
      'ai' => ['enabled' => false],
    ],
  ]);
}

// --- Heuristic checks
$issues = [];
$normToIds = [];

foreach ($rows as $r) {
  $qid = (string)($r['id'] ?? '');
  $prompt = trim((string)($r['prompt'] ?? ''));
  $pPrev = qa_preview($prompt);

  if ($prompt === '' || (function_exists('mb_strlen') ? mb_strlen($prompt, 'UTF-8') : strlen($prompt)) < 8) {
    qa_add_issue($issues, $qid, 'high', 'PROMPT_TOO_SHORT', 'Question prompt is missing or too short.', $pPrev);
  }

  $norm = qa_norm($prompt);
  if ($norm !== '') {
    if (!isset($normToIds[$norm])) $normToIds[$norm] = [];
    $normToIds[$norm][] = $qid;
  }

  if ($isTheory) {
    $guide = trim((string)($r['guide'] ?? ''));
    if ($guide === '') {
      qa_add_issue($issues, $qid, 'medium', 'MISSING_GUIDE', 'Missing theory marking guide (guide).', $pPrev);
    }

    $pts = [];
    $ptsRaw = $r['pointsJson'] ?? null;
    if ($ptsRaw !== null && $ptsRaw !== '') {
      $d = json_decode((string)$ptsRaw, true);
      if (is_array($d)) {
        foreach ($d as $p) {
          $s = trim((string)$p);
          if ($s !== '') $pts[] = $s;
        }
      }
    }
    if (count($pts) === 0) {
      qa_add_issue($issues, $qid, 'medium', 'MISSING_POINTS', 'Missing theory key points / rubric (points_json).', $pPrev);
    } elseif (count($pts) < 4) {
      qa_add_issue($issues, $qid, 'low', 'TOO_FEW_POINTS', 'Very few key points; rubric may be too thin.', $pPrev, ['pointsCount' => count($pts)]);
    }
  } else {
    // MCQ checks
    $optsRaw = $r['options'] ?? null;
    $opts = is_string($optsRaw) ? json_decode($optsRaw, true) : null;
    if (!is_array($opts)) {
      $opts = [];
      qa_add_issue($issues, $qid, 'high', 'OPTIONS_INVALID', 'MCQ options are missing or invalid JSON.', $pPrev);
    }
    $optsClean = [];
    foreach ($opts as $o) {
      $s = trim((string)$o);
      if ($s === '') {
        qa_add_issue($issues, $qid, 'low', 'EMPTY_OPTION', 'An option is empty/blank.', $pPrev);
        continue;
      }
      $optsClean[] = $s;
    }

    if (count($optsClean) < 2) {
      qa_add_issue($issues, $qid, 'high', 'OPTIONS_TOO_FEW', 'MCQ has fewer than 2 valid options.', $pPrev, ['optionsCount' => count($optsClean)]);
    }

    // Duplicate options
    $seen = [];
    $dups = [];
    foreach ($optsClean as $o) {
      $n = qa_norm($o);
      if ($n === '') continue;
      if (isset($seen[$n])) $dups[$n] = true;
      $seen[$n] = true;
    }
    if (count($dups) > 0) {
      qa_add_issue($issues, $qid, 'medium', 'DUPLICATE_OPTIONS', 'Some MCQ options look duplicated.', $pPrev);
    }

    $answerIndex = $r['answerIndex'] ?? null;
    if ($answerIndex === null || $answerIndex === '') {
      qa_add_issue($issues, $qid, 'high', 'MISSING_ANSWER_INDEX', 'MCQ answer_index is missing.', $pPrev);
    } else {
      $ai = (int)$answerIndex;
      if ($ai < 0 || $ai >= count($optsClean)) {
        qa_add_issue($issues, $qid, 'high', 'ANSWER_INDEX_OOR', 'MCQ answer_index is out of range for options.', $pPrev, ['answerIndex' => $ai, 'optionsCount' => count($optsClean)]);
      }
    }

    $ex = trim((string)($r['explanation'] ?? ''));
    if ($ex === '') {
      qa_add_issue($issues, $qid, 'low', 'MISSING_EXPLANATION', 'MCQ explanation is missing.', $pPrev);
    }
  }
}

// Duplicate prompts (heuristic)
foreach ($normToIds as $norm => $ids) {
  if (!is_array($ids) || count($ids) < 2) continue;
  // Mark all but the first as duplicates
  $first = true;
  foreach ($ids as $qid) {
    if ($first) {
      $first = false;
      continue;
    }
    qa_add_issue($issues, (string)$qid, 'medium', 'DUPLICATE_PROMPT', 'This prompt appears duplicated in the bank.', '');
  }
}

// Summary counts
$qWithIssues = [];
foreach ($issues as $is) {
  $qid = (string)($is['questionId'] ?? '');
  if ($qid !== '') $qWithIssues[$qid] = true;
}

$aiOut = ['enabled' => false];

// --- Optional AI audit
if ($includeAi) {
  $aiOut = ['enabled' => true, 'model' => null, 'reviewedCount' => 0, 'items' => []];

  // Pick a subset: prioritize questions that already have heuristic issues.
  $pickIds = array_keys($qWithIssues);
  $picked = [];
  $byId = [];
  foreach ($rows as $r) {
    $byId[(string)($r['id'] ?? '')] = $r;
  }

  foreach ($pickIds as $qid) {
    if (isset($byId[$qid])) $picked[] = $byId[$qid];
    if (count($picked) >= $aiLimit) break;
  }
  if (count($picked) < $aiLimit) {
    foreach ($rows as $r) {
      if (count($picked) >= $aiLimit) break;
      $qid = (string)($r['id'] ?? '');
      if ($qid === '') continue;
      // Skip ones already picked
      $already = false;
      foreach ($picked as $pr) {
        if ((string)($pr['id'] ?? '') === $qid) { $already = true; break; }
      }
      if (!$already) $picked[] = $r;
    }
  }

  $payload = [];
  foreach ($picked as $r) {
    $qid = (string)($r['id'] ?? '');
    $prompt = trim((string)($r['prompt'] ?? ''));
    if ($qid === '' || $prompt === '') continue;

    if ($isTheory) {
      $pts = [];
      $ptsRaw = $r['pointsJson'] ?? null;
      if ($ptsRaw !== null && $ptsRaw !== '') {
        $d = json_decode((string)$ptsRaw, true);
        if (is_array($d)) {
          foreach ($d as $p) {
            $s = trim((string)$p);
            if ($s !== '') $pts[] = $s;
          }
        }
      }
      $payload[] = [
        'id' => $qid,
        'prompt' => $prompt,
        'guide' => trim((string)($r['guide'] ?? '')),
        'points' => $pts,
      ];
    } else {
      $optsRaw = $r['options'] ?? null;
      $opts = is_string($optsRaw) ? json_decode($optsRaw, true) : null;
      if (!is_array($opts)) $opts = [];
      $optsClean = [];
      foreach ($opts as $o) {
        $s = trim((string)$o);
        if ($s !== '') $optsClean[] = $s;
      }
      $payload[] = [
        'id' => $qid,
        'prompt' => $prompt,
        'options' => $optsClean,
        'answerIndex' => $r['answerIndex'] ?? null,
        'explanation' => trim((string)($r['explanation'] ?? '')),
      ];
    }
  }

  if (count($payload) > 0) {
    ai_enforce_limits($pdo, (string)$me['id'], 'quality_audit');
    $callHash = hash('sha256', json_encode([
      'bankId' => $bankId,
      'mode' => $mode,
      'reviewCount' => count($payload),
    ], JSON_UNESCAPED_SLASHES));
    ai_log_call($pdo, (string)$me['id'], 'quality_audit', $callHash);

    $courseCode = (string)($bank['courseCode'] ?? $bank['courseId'] ?? '');
    $courseTitle = (string)($bank['courseTitle'] ?? '');
    $bankTitle = (string)($bank['title'] ?? '');

    $aiPrompt = "You are a question bank quality auditor for Nigerian university study materials.\n"
      . "Safety rules: do not include hate/harassment/sexual content, do not provide illegal instructions, and do not request personal data.\n"
      . "Focus on STRUCTURAL quality: clarity, ambiguity, grammar, formatting, internal consistency (options/answerIndex/explanation), duplicates.\n"
      . "Do NOT claim factual correctness unless you are certain; if unsure, say 'uncertain'.\n\n"
      . "Return ONLY valid JSON (no markdown, no extra text) in this exact shape:\n"
      . '{"items":[{"id":"...","severity":"low","issues":["..."],"suggestedFix":"..."}]}' . "\n\n"
      . "Rules:\n"
      . "- severity: low|medium|high\n"
      . "- issues: 1-6 short strings\n"
      . "- suggestedFix: <= 2 sentences (or empty string)\n\n"
      . "Course: {$courseCode}" . ($courseTitle !== '' ? " ({$courseTitle})" : '') . "\n"
      . "Bank: {$bankTitle}\n"
      . "Mode: " . ($isTheory ? 'theory' : 'mcq') . "\n\n"
      . "QUESTIONS JSON:\n" . json_encode(['questions' => $payload], JSON_UNESCAPED_SLASHES);

    $resp = gemini_generate_text($aiPrompt, [
      'temperature' => 0.2,
      'maxOutputTokens' => 1800,
    ]);

    if ($resp['ok']) {
      $t = trim((string)($resp['text'] ?? ''));
      $j = ai_extract_json($t);
      $aiData = $j !== null ? json_decode($j, true) : null;
      if (is_array($aiData) && isset($aiData['items']) && is_array($aiData['items'])) {
        $itemsOut = [];
        foreach ($aiData['items'] as $it) {
          if (!is_array($it)) continue;
          $id = trim((string)($it['id'] ?? ''));
          if ($id === '') continue;
          $sev = strtolower(trim((string)($it['severity'] ?? 'low')));
          if (!in_array($sev, ['low','medium','high'], true)) $sev = 'low';
          $iss = $it['issues'] ?? [];
          if (!is_array($iss)) $iss = [];
          $issOut = [];
          foreach ($iss as $s) {
            $s = trim((string)$s);
            if ($s !== '') $issOut[] = $s;
            if (count($issOut) >= 6) break;
          }
          $fix = trim((string)($it['suggestedFix'] ?? ''));
          if (function_exists('mb_strlen') && mb_strlen($fix, 'UTF-8') > 400) {
            $fix = (string)mb_substr($fix, 0, 400, 'UTF-8') . '…';
          } elseif (strlen($fix) > 400) {
            $fix = substr($fix, 0, 400) . '…';
          }
          $itemsOut[] = [
            'questionId' => $id,
            'severity' => $sev,
            'issues' => $issOut,
            'suggestedFix' => $fix,
          ];
        }
        $aiOut['model'] = gemini_model();
        $aiOut['reviewedCount'] = count($payload);
        $aiOut['items'] = $itemsOut;
      } else {
        $aiOut['model'] = gemini_model();
        $aiOut['reviewedCount'] = count($payload);
        $aiOut['items'] = [];
        $aiOut['warning'] = 'AI audit returned an unexpected format.';
      }
    } else {
      $aiOut['model'] = gemini_model();
      $aiOut['reviewedCount'] = count($payload);
      $aiOut['items'] = [];
      $aiOut['warning'] = $resp['error'] ?? 'AI audit failed.';
    }
  }
}

json_response([
  'success' => true,
  'data' => [
    'bank' => [
      'id' => (string)$bank['id'],
      'title' => (string)($bank['title'] ?? ''),
      'courseId' => (string)($bank['courseId'] ?? ''),
      'mode' => $mode,
    ],
    'summary' => [
      'totalQuestions' => count($rows),
      'heuristicIssueCount' => count($issues),
      'questionsWithIssues' => count($qWithIssues),
      'aiReviewedCount' => (int)($aiOut['reviewedCount'] ?? 0),
      'aiItemsCount' => isset($aiOut['items']) && is_array($aiOut['items']) ? count($aiOut['items']) : 0,
    ],
    'issues' => $issues,
    'ai' => $aiOut,
  ],
]);
