<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

require_once __DIR__ . '/../ai/_gemini.php';
require_once __DIR__ . '/../ai/_limits.php';
require_once __DIR__ . '/../ai/_cache.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
require_admin($me);

$body = read_json();
require_fields($body, ['bankId']);

$bankId = trim((string)$body['bankId']);
$limit = (int)($body['limit'] ?? 30);
$useAi = (bool)($body['useAi'] ?? false);

if ($limit < 1) $limit = 1;
if ($limit > 80) $limit = 80;

// Fetch bank metadata
$b = $pdo->prepare('SELECT b.id, b.title, b.mode, b.course_id AS courseId, c.code, c.title AS courseTitle
                    FROM banks b
                    LEFT JOIN courses c ON c.id = b.course_id
                    WHERE b.id = :id LIMIT 1');
$b->execute([':id' => $bankId]);
$bank = $b->fetch();
if (!$bank) json_response(['success' => false, 'error' => 'Bank not found'], 404);

$mode = strtolower((string)($bank['mode'] ?? 'mcq'));
if ($mode !== 'theory') {
  json_response(['success' => false, 'error' => 'This bank is not a theory bank'], 422);
}

// Load questions
try {
  $q = $pdo->prepare('SELECT id, prompt, guide, points_json AS pointsJson, sort_order AS sortOrder
                      FROM theory_questions
                      WHERE bank_id = :bid
                      ORDER BY sort_order ASC, id ASC
                      LIMIT ' . (int)$limit);
  $q->execute([':bid' => $bankId]);
  $rows = $q->fetchAll() ?: [];
} catch (Throwable $e) {
  json_response(['success' => false, 'error' => 'Theory questions table not available. Run the theory migration first.'], 409);
}

if (count($rows) === 0) {
  json_response(['success' => true, 'data' => ['items' => [], 'summary' => ['total' => 0]]]);
}

// -----------------------------
// Heuristic audit (cheap, no AI)
// -----------------------------

function tnorm(string $s): string {
  $s = strtolower(trim($s));
  $s = preg_replace('/\s+/u', ' ', $s) ?? $s;
  $s = preg_replace('/[^a-z0-9 ]+/u', '', $s) ?? $s;
  return trim($s);
}

// Detect duplicate prompts (within this result set)
$promptBuckets = [];
foreach ($rows as $r) {
  $p = tnorm((string)($r['prompt'] ?? ''));
  if ($p === '') continue;
  $promptBuckets[$p] = $promptBuckets[$p] ?? [];
  $promptBuckets[$p][] = (string)($r['id'] ?? '');
}
$dupIds = [];
foreach ($promptBuckets as $k => $ids) {
  if (count($ids) > 1) {
    foreach ($ids as $id) $dupIds[$id] = true;
  }
}

$items = [];
$summary = [
  'total' => count($rows),
  'missingGuide' => 0,
  'missingPoints' => 0,
  'promptTooShort' => 0,
  'guideTooShort' => 0,
  'tooFewPoints' => 0,
  'tooManyPoints' => 0,
  'duplicates' => 0,
];

foreach ($rows as $r) {
  $id = (string)($r['id'] ?? '');
  $prompt = (string)($r['prompt'] ?? '');
  $guide = $r['guide'] === null ? '' : (string)$r['guide'];

  $points = [];
  $raw = $r['pointsJson'] ?? null;
  if ($raw !== null && $raw !== '') {
    $d = json_decode((string)$raw, true);
    if (is_array($d)) {
      foreach ($d as $p) {
        $p = trim((string)$p);
        if ($p !== '') $points[] = $p;
      }
    }
  }

  $issues = [];

  $pLen = mb_strlen(trim($prompt));
  $gLen = mb_strlen(trim($guide));
  $ptCount = count($points);

  if (trim($guide) === '') {
    $issues[] = 'MISSING_GUIDE';
    $summary['missingGuide']++;
  } else if ($gLen < 60) {
    $issues[] = 'GUIDE_TOO_SHORT';
    $summary['guideTooShort']++;
  }

  if ($raw === null || $ptCount === 0) {
    $issues[] = 'MISSING_POINTS';
    $summary['missingPoints']++;
  } else {
    if ($ptCount < 3) {
      $issues[] = 'TOO_FEW_POINTS';
      $summary['tooFewPoints']++;
    }
    if ($ptCount > 12) {
      $issues[] = 'TOO_MANY_POINTS';
      $summary['tooManyPoints']++;
    }
  }

  if ($pLen < 30) {
    $issues[] = 'PROMPT_TOO_SHORT';
    $summary['promptTooShort']++;
  }

  if (isset($dupIds[$id])) {
    $issues[] = 'DUPLICATE_PROMPT';
    $summary['duplicates']++;
  }

  $severity = 'low';
  if (in_array('MISSING_GUIDE', $issues, true) || in_array('MISSING_POINTS', $issues, true)) $severity = 'high';
  else if (in_array('PROMPT_TOO_SHORT', $issues, true) || in_array('GUIDE_TOO_SHORT', $issues, true)) $severity = 'medium';

  $preview = trim(preg_replace('/\s+/u', ' ', $prompt) ?? $prompt);
  if (mb_strlen($preview) > 140) $preview = mb_substr($preview, 0, 140) . '…';

  $items[] = [
    'id' => $id,
    'promptPreview' => $preview,
    'issues' => $issues,
    'severity' => $severity,
    'stats' => [
      'promptChars' => $pLen,
      'guideChars' => $gLen,
      'pointsCount' => $ptCount,
      'sortOrder' => (int)($r['sortOrder'] ?? 1),
    ],
  ];
}

// -----------------------------
// Optional AI audit (batched)
// -----------------------------

$aiItems = [];
$aiUsed = false;

if ($useAi) {
  // Cache key based on content hash so repeated audits don't re-bill.
  $contentHash = hash('sha256', json_encode(array_map(static function ($r) {
    return [
      'id' => (string)($r['id'] ?? ''),
      'prompt' => (string)($r['prompt'] ?? ''),
      'guide' => (string)($r['guide'] ?? ''),
      'pointsJson' => (string)($r['pointsJson'] ?? ''),
    ];
  }, $rows), JSON_UNESCAPED_SLASHES));

  $cacheKey = ai_cache_key('theory_audit', [
    'bankId' => $bankId,
    'limit' => $limit,
    'contentHash' => $contentHash,
    'model' => gemini_model(),
  ]);

  $cachedJson = ai_cache_get($pdo, $cacheKey);
  if (is_string($cachedJson)) {
    $tmp = json_decode($cachedJson, true);
    if (is_array($tmp) && isset($tmp['items']) && is_array($tmp['items'])) {
      $aiItems = $tmp['items'];
      $aiUsed = true;
    }
  }

  if (!$aiUsed) {
    // Limit the AI audit payload so prompts stay small.
    $maxAi = min(12, count($rows));
    $payload = [];
    for ($i = 0; $i < $maxAi; $i++) {
      $r = $rows[$i];
      $pts = [];
      $raw = $r['pointsJson'] ?? null;
      if ($raw !== null && $raw !== '') {
        $d = json_decode((string)$raw, true);
        if (is_array($d)) $pts = array_values(array_slice($d, 0, 10));
      }

      $payload[] = [
        'id' => (string)($r['id'] ?? ''),
        'prompt' => (string)($r['prompt'] ?? ''),
        'guide' => $r['guide'] === null ? '' : (string)$r['guide'],
        'points' => $pts,
      ];
    }

    ai_enforce_limits($pdo, (string)$me['id'], 'audit_theory');
    $callHash = hash('sha256', json_encode(['bankId' => $bankId, 'contentHash' => $contentHash, 'n' => count($payload)], JSON_UNESCAPED_SLASHES));
    ai_log_call($pdo, (string)$me['id'], 'audit_theory', $callHash);

    $courseCode = (string)($bank['code'] ?? $bank['courseId'] ?? '');
    $courseTitle = (string)($bank['courseTitle'] ?? '');
    $bankTitle = (string)($bank['title'] ?? '');

    $prompt = "You are a strict but fair content quality auditor for Nigerian university theory questions.\n"
      . "Safety rules: do not include hate/harassment/sexual content, do not provide illegal instructions, and do not request personal data.\n"
      . "Task: audit each theory question and provide clarity + marking-scheme quality feedback.\n"
      . "Course: {$courseCode}" . ($courseTitle !== '' ? " ({$courseTitle})" : '') . "\n"
      . "Bank: {$bankTitle}\n\n"
      . "Return ONLY valid JSON (no markdown, no extra text) in this exact shape:\n"
      . "{\"items\":[{\"id\":\"...\",\"clarityScore\":4,\"issues\":[\"...\"],\"suggestions\":[\"...\"]}]}\n\n"
      . "Rules:\n"
      . "- clarityScore is integer 1..5 (5 = very clear, 1 = confusing).\n"
      . "- issues: 1-5 short issue labels (e.g., 'too broad', 'missing keywords', 'unclear scope', 'marking points too vague').\n"
      . "- suggestions: 2-6 actionable edits.\n"
      . "- Do NOT rewrite full answers; just audit the question, guide, and points.\n\n"
      . "INPUT JSON:\n" . json_encode(['questions' => $payload], JSON_UNESCAPED_SLASHES);

    $resp = gemini_generate_text($prompt, [
      'temperature' => 0.2,
      'maxOutputTokens' => 1400,
    ]);

    if (!$resp['ok']) {
      json_response(['success' => false, 'error' => $resp['error'] ?? 'AI failed'], (int)($resp['status'] ?? 502));
    }

    $text = trim((string)$resp['text']);
    $jsonText = ai_extract_json($text);
    $out = $jsonText !== null ? json_decode($jsonText, true) : null;

    if (!is_array($out) || !isset($out['items']) || !is_array($out['items'])) {
      json_response(['success' => false, 'error' => 'AI returned invalid format'], 502);
    }

    $aiItems = $out['items'];
    $aiUsed = true;

    // Cache for 7 days
    ai_cache_set(
      $pdo,
      $cacheKey,
      'theory_audit',
      gemini_model(),
      json_encode(['items' => $aiItems, 'model' => gemini_model()], JSON_UNESCAPED_SLASHES),
      60 * 60 * 24 * 7
    );
  }

  // Merge AI findings onto heuristic items
  $aiMap = [];
  foreach ($aiItems as $it) {
    if (!is_array($it)) continue;
    $id = (string)($it['id'] ?? '');
    if ($id === '') continue;
    $aiMap[$id] = [
      'clarityScore' => (int)($it['clarityScore'] ?? 0),
      'issues' => is_array($it['issues'] ?? null) ? array_values(array_filter(array_map('strval', (array)$it['issues']))) : [],
      'suggestions' => is_array($it['suggestions'] ?? null) ? array_values(array_filter(array_map('strval', (array)$it['suggestions']))) : [],
    ];
  }

  foreach ($items as &$it) {
    $id = (string)($it['id'] ?? '');
    if ($id !== '' && isset($aiMap[$id])) {
      $it['ai'] = $aiMap[$id];
    }
  }
  unset($it);
}

json_response([
  'success' => true,
  'data' => [
    'bank' => [
      'id' => (string)$bankId,
      'title' => (string)($bank['title'] ?? ''),
      'courseId' => (string)($bank['courseId'] ?? ''),
      'courseCode' => (string)($bank['code'] ?? ''),
    ],
    'summary' => $summary,
    'items' => $items,
    'aiUsed' => $useAi,
    'model' => $useAi ? gemini_model() : null,
  ],
]);
