<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
require_admin($me);

$body = read_json();
// Backward compatible payload:
// - new: { userId, uploadsDisabled: boolean }
// - old: { userId, disabled: boolean }
require_fields($body, ['userId']);

$userId = trim((string)$body['userId']);

$uploadsDisabledRaw = null;
if (array_key_exists('uploadsDisabled', $body)) {
  $uploadsDisabledRaw = $body['uploadsDisabled'];
} elseif (array_key_exists('disabled', $body)) {
  $uploadsDisabledRaw = $body['disabled'];
}

if ($uploadsDisabledRaw === null) {
  json_response(['success' => false, 'error' => 'Missing uploadsDisabled'], 422);
}

$uploadsDisabled = (int)!!$uploadsDisabledRaw;

if ($userId === '') {
  json_response(['success' => false, 'error' => 'Missing userId'], 422);
}

try {
  ensure_course_rep_flags_table($pdo);
  $stmt = $pdo->prepare('INSERT INTO course_rep_flags (user_id, uploads_disabled, updated_at)
                         VALUES (:uid,:d,UTC_TIMESTAMP())
                         ON DUPLICATE KEY UPDATE uploads_disabled = VALUES(uploads_disabled), updated_at = UTC_TIMESTAMP()');
  $stmt->execute([':uid' => $userId, ':d' => $uploadsDisabled]);
} catch (Throwable $e) {
  json_response(['success' => false, 'error' => 'Failed to update flag'], 500);
}

upload_audit_log($pdo, [
  'action' => $uploadsDisabled ? 'disable_uploads' : 'enable_uploads',
  'entityType' => 'course_rep_flag',
  'entityId' => $userId,
  'actorId' => (string)$me['id'],
  'targetUserId' => $userId,
  'details' => ['uploadsDisabled' => $uploadsDisabled]
]);

json_response(['success' => true, 'data' => ['updated' => true, 'uploadsDisabled' => $uploadsDisabled === 1]]);
