<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'GET') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
require_admin($me);

$status = isset($_GET['status']) ? trim((string)$_GET['status']) : 'pending';
if (!in_array($status, ['pending','approved','denied'], true)) {
  $status = 'pending';
}

// Aggregate course details per request
$sql = 'SELECT r.id, r.user_id, r.status, r.level, r.reason, r.proof_url, r.requested_at, r.reviewed_at, r.reviewed_by, r.admin_note,
               u.email, u.full_name,
               GROUP_CONCAT(c.id ORDER BY c.level, c.code SEPARATOR ",") AS course_ids,
               GROUP_CONCAT(CONCAT(c.code, " — ", c.title, " (", c.level, ")") ORDER BY c.level, c.code SEPARATOR "||") AS course_labels
        FROM course_rep_requests r
        JOIN users u ON u.id = r.user_id
        LEFT JOIN course_rep_request_courses rc ON rc.request_id = r.id
        LEFT JOIN courses c ON c.id = rc.course_id
        WHERE r.status = :st
        GROUP BY r.id
        ORDER BY r.requested_at DESC';

try {
  $stmt = $pdo->prepare($sql);
  $stmt->execute([':st' => $status]);
  $rows = $stmt->fetchAll() ?: [];
} catch (PDOException $e) {
  // Common first-run issue: migration not applied yet
  if (($e->getCode() ?? '') === '42S02') {
    json_response([
      'success' => false,
      'error' => 'Missing course rep tables. Run the course_rep migration SQL to create course_rep_requests and related tables.'
    ], 500);
  }
  throw $e;
}

$out = [];
foreach ($rows as $row) {
  $courseIds = $row['course_ids'] ? array_values(array_filter(explode(',', (string)$row['course_ids']))) : [];
  $courseLabels = $row['course_labels'] ? array_values(array_filter(explode('||', (string)$row['course_labels']))) : [];
  $courses = [];
  for ($i = 0; $i < count($courseIds); $i++) {
    $courses[] = [
      'id' => $courseIds[$i],
      'label' => $courseLabels[$i] ?? $courseIds[$i]
    ];
  }

  $out[] = [
    'id' => $row['id'],
    'status' => $row['status'],
    'level' => $row['level'] !== null ? (int)$row['level'] : null,
    'reason' => $row['reason'],
    'proofUrl' => $row['proof_url'],
    'requestedAt' => $row['requested_at'],
    'reviewedAt' => $row['reviewed_at'],
    'adminNote' => $row['admin_note'],
    'user' => [
      'id' => $row['user_id'],
      'email' => $row['email'],
      'fullName' => $row['full_name'],
    ],
    'courses' => $courses,
  ];
}

json_response(['success' => true, 'data' => ['requests' => $out]]);
