<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
require_admin($me);

$body = read_json();
require_fields($body, ['requestId']);

$requestId = trim((string)$body['requestId']);
$overrideCourseIds = $body['courseIds'] ?? null;
$adminNote = trim((string)($body['adminNote'] ?? ''));

if ($requestId === '') {
  json_response(['success' => false, 'error' => 'requestId is required'], 422);
}

try {
  $pdo->beginTransaction();

  // Lock request row
  $stmt = $pdo->prepare('SELECT * FROM course_rep_requests WHERE id = :id LIMIT 1 FOR UPDATE');
  $stmt->execute([':id' => $requestId]);
  $req = $stmt->fetch();
  if (!$req) {
    json_response(['success' => false, 'error' => 'Request not found'], 404);
  }
  if ((string)$req['status'] !== 'pending') {
    json_response(['success' => false, 'error' => 'Request is not pending'], 409);
  }

  $courseIds = [];
  if (is_array($overrideCourseIds)) {
    $courseIds = array_values(array_unique(array_filter(array_map(fn($c) => trim((string)$c), $overrideCourseIds))));
  } else {
    $stmt = $pdo->prepare('SELECT course_id FROM course_rep_request_courses WHERE request_id = :rid');
    $stmt->execute([':rid' => $requestId]);
    $courseIds = array_values(array_map(fn($r) => $r['course_id'], $stmt->fetchAll() ?: []));
  }
  if (count($courseIds) < 1) {
    json_response(['success' => false, 'error' => 'No courses selected for this approval'], 422);
  }

  // Approve request
  $pdo->prepare('UPDATE course_rep_requests
                 SET status = "approved", reviewed_at = UTC_TIMESTAMP(), reviewed_by = :by, admin_note = :note
                 WHERE id = :id')
      ->execute([':by' => $me['id'], ':note' => ($adminNote !== '' ? $adminNote : null), ':id' => $requestId]);

  // Promote user to course_rep unless already admin
  $pdo->prepare('UPDATE users SET role = IF(role = "admin", "admin", "course_rep") WHERE id = :uid')
      ->execute([':uid' => $req['user_id']]);

  // Assign upload permissions
  $ins = $pdo->prepare('INSERT IGNORE INTO course_rep_courses (user_id, course_id, assigned_by, assigned_at)
                        VALUES (:uid,:cid,:by,UTC_TIMESTAMP())');
  foreach ($courseIds as $cid) {
    $ins->execute([':uid' => $req['user_id'], ':cid' => $cid, ':by' => $me['id']]);
  }

  // Return updated user
  $stmt = $pdo->prepare('SELECT * FROM users WHERE id = :uid LIMIT 1');
  $stmt->execute([':uid' => $req['user_id']]);
  $u = $stmt->fetch();
  $safe = $u ? safe_user($pdo, $u) : null;

  $pdo->commit();

  json_response(['success' => true, 'data' => [
    'approved' => true,
    'user' => $safe,
    'courseIds' => $courseIds,
  ]]);
} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  json_response(['success' => false, 'error' => 'Failed to approve request'], 500);
}
