<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'GET') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
require_admin($me);

$q = isset($_GET['q']) ? trim((string)$_GET['q']) : '';
$courseId = isset($_GET['courseId']) ? trim((string)$_GET['courseId']) : '';
$entityType = isset($_GET['entityType']) ? trim((string)$_GET['entityType']) : '';
$action = isset($_GET['action']) ? trim((string)$_GET['action']) : '';
$limit = isset($_GET['limit']) ? max(1, min(200, (int)$_GET['limit'])) : 100;

try {
  ensure_upload_audit_table($pdo);
} catch (Throwable $e) {
  // If the DB doesn't support JSON, still try to return empty.
}

$where = ['1=1'];
$params = [];

if ($courseId !== '') {
  $where[] = 'a.course_id = :cid';
  $params[':cid'] = $courseId;
}
if ($entityType !== '') {
  $where[] = 'a.entity_type = :et';
  $params[':et'] = $entityType;
}
if ($action !== '') {
  $where[] = 'a.action = :act';
  $params[':act'] = $action;
}
if ($q !== '') {
  // search: actor/target name/email OR entity id
  $where[] = '(a.entity_id LIKE :q OR a.file_path LIKE :q OR u1.full_name LIKE :q OR u1.email LIKE :q OR u2.full_name LIKE :q OR u2.email LIKE :q)';
  $params[':q'] = '%' . $q . '%';
}

$sql = 'SELECT a.id, a.action, a.entity_type AS entityType, a.entity_id AS entityId, a.course_id AS courseId, a.file_path AS filePath,
               a.details_json AS details,
               a.created_at AS createdAt,
               u1.id AS actorId, u1.full_name AS actorName, u1.email AS actorEmail,
               u2.id AS targetUserId, u2.full_name AS targetName, u2.email AS targetEmail
        FROM upload_audit a
        LEFT JOIN users u1 ON u1.id = a.actor_id
        LEFT JOIN users u2 ON u2.id = a.target_user_id
        WHERE ' . implode(' AND ', $where) . '
        ORDER BY a.created_at DESC
        LIMIT ' . (int)$limit;

try {
  $stmt = $pdo->prepare($sql);
  $stmt->execute($params);
  $rows = $stmt->fetchAll() ?: [];
} catch (Throwable $e) {
  // if table doesn't exist or query fails, return empty
  $rows = [];
}

$logs = array_map(function($r){
  $details = null;
  if (isset($r['details'])) {
    $details = json_get($r['details'], null);
  }
  return [
    'id' => $r['id'] ?? null,
    'action' => $r['action'] ?? null,
    'entityType' => $r['entityType'] ?? null,
    'entityId' => $r['entityId'] ?? null,
    'courseId' => $r['courseId'] ?? null,
    'filePath' => $r['filePath'] ?? null,
    'createdAt' => $r['createdAt'] ?? null,
    'actor' => [
      'id' => $r['actorId'] ?? null,
      'fullName' => $r['actorName'] ?? null,
      'email' => $r['actorEmail'] ?? null,
    ],
    'target' => [
      'id' => $r['targetUserId'] ?? null,
      'fullName' => $r['targetName'] ?? null,
      'email' => $r['targetEmail'] ?? null,
    ],
    'details' => $details,
  ];
}, $rows);

json_response(['success' => true, 'data' => ['logs' => $logs]]);
