<?php
declare(strict_types=1);

/**
 * Simple DB-backed AI response cache.
 *
 * Table: ai_cache
 * - Cache key is SHA-256 hex (64 chars)
 * - We store response_json as TEXT/MEDIUMTEXT
 *
 * Important: cache hits should NOT count against ai_request_log quotas.
 */

/**
 * Build a stable cache key from a purpose + normalized parameters.
 */
function ai_cache_key(string $purpose, array $params): string {
  ksort($params);
  return hash('sha256', json_encode(['purpose' => $purpose, 'params' => $params], JSON_UNESCAPED_SLASHES));
}

/**
 * Get cached JSON string or null.
 * Degrades gracefully if table doesn't exist.
 */
function ai_cache_get(PDO $pdo, string $cacheKey): ?string {
  try {
    $st = $pdo->prepare(
      'SELECT response_json FROM ai_cache
       WHERE cache_key = :k
         AND (expires_at IS NULL OR expires_at > UTC_TIMESTAMP())
       LIMIT 1'
    );
    $st->execute([':k' => $cacheKey]);
    $row = $st->fetch();
    if (!$row) return null;

    // best-effort hit stats
    try {
      $pdo->prepare('UPDATE ai_cache SET hit_count = hit_count + 1, last_hit = UTC_TIMESTAMP() WHERE cache_key = :k')
          ->execute([':k' => $cacheKey]);
    } catch (Throwable $e) {
      // ignore
    }

    $val = (string)($row['response_json'] ?? '');
    return trim($val) !== '' ? $val : null;
  } catch (Throwable $e) {
    return null;
  }
}

/**
 * Save a cached JSON string.
 * Degrades gracefully if table doesn't exist.
 */
function ai_cache_set(PDO $pdo, string $cacheKey, string $purpose, string $model, string $responseJson, ?int $ttlSeconds = 86400): void {
  try {
    $expiresAt = $ttlSeconds === null ? null : gmdate('Y-m-d H:i:s', time() + $ttlSeconds);
    $st = $pdo->prepare(
      'INSERT INTO ai_cache (cache_key, purpose, model, response_json, created_at, expires_at)
       VALUES (:k, :p, :m, :r, UTC_TIMESTAMP(), :e)
       ON DUPLICATE KEY UPDATE
         response_json = VALUES(response_json),
         model = VALUES(model),
         expires_at = VALUES(expires_at)'
    );
    $st->execute([
      ':k' => $cacheKey,
      ':p' => $purpose,
      ':m' => $model,
      ':r' => $responseJson,
      ':e' => $expiresAt,
    ]);
  } catch (Throwable $e) {
    // ignore
  }
}
