<?php
declare(strict_types=1);

/**
 * Best-effort text extraction for PDFs (for Ask-AI over materials/past questions).
 * Works only if `pdftotext` is available and shell_exec is enabled on the host.
 *
 * If extraction isn't possible, return null.
 */

/**
 * Extract text and return a small debug reason if it fails.
 *
 * Reasons:
 * - OK
 * - FILE_MISSING
 * - SHELL_EXEC_DISABLED
 * - PDFTOTEXT_MISSING
 * - EMPTY_OUTPUT
 * - LIKELY_SCANNED
 */
function pdf_extract_text_with_reason(string $absPath): array {
  if (!is_file($absPath)) {
    return ['text' => null, 'reason' => 'FILE_MISSING'];
  }

  // shell_exec might be disabled in shared hosting
  if (!function_exists('shell_exec')) {
    return ['text' => null, 'reason' => 'SHELL_EXEC_DISABLED'];
  }

  $bin = trim((string)@shell_exec('command -v pdftotext 2>/dev/null'));
  if ($bin === '') {
    return ['text' => null, 'reason' => 'PDFTOTEXT_MISSING'];
  }

  $tmp = tempnam(sys_get_temp_dir(), 'pdftext_');
  if (!$tmp) {
    return ['text' => null, 'reason' => 'EMPTY_OUTPUT'];
  }

  $cmd = escapeshellcmd($bin) . ' -layout -enc UTF-8 ' . escapeshellarg($absPath) . ' ' . escapeshellarg($tmp) . ' 2>/dev/null';
  @shell_exec($cmd);

  $txt = @file_get_contents($tmp);
  @unlink($tmp);

  if (!is_string($txt)) {
    return ['text' => null, 'reason' => 'EMPTY_OUTPUT'];
  }

  $txt = trim($txt);
  if ($txt === '') {
    return ['text' => null, 'reason' => 'EMPTY_OUTPUT'];
  }

  // If too short, likely scanned PDF (no embedded text)
  if (mb_strlen($txt) < 40) {
    return ['text' => null, 'reason' => 'LIKELY_SCANNED'];
  }

  // Hard cap context size (keep memory and prompt size sane)
  if (strlen($txt) > 200000) {
    $txt = substr($txt, 0, 200000);
  }

  return ['text' => $txt, 'reason' => 'OK'];
}

function pdf_to_text_best_effort(string $absPath): ?string {
  $out = pdf_extract_text_with_reason($absPath);
  return is_string($out['text'] ?? null) ? (string)$out['text'] : null;
}

/**
 * Safe join to map a DB file_path like "/uploads/x.pdf" to an absolute path.
 * Returns null if path escapes the API root.
 */
function resolve_api_file_path(string $apiRootAbs, string $filePath): ?string {
  $filePath = (string)$filePath;
  if ($filePath === '') return null;

  // Ensure leading slash for consistency
  if ($filePath[0] !== '/') $filePath = '/' . $filePath;

  $candidate = realpath($apiRootAbs . $filePath);
  if ($candidate === false) return null;

  // Prevent traversal
  $apiRootAbs = rtrim($apiRootAbs, DIRECTORY_SEPARATOR);
  if (strpos($candidate, $apiRootAbs) !== 0) return null;

  return $candidate;
}
