<?php
declare(strict_types=1);

/**
 * Minimal Gemini (Google Generative Language API) client with key failover.
 *
 * Configure on server (recommended):
 *   GEMINI_API_KEYS="KEY1,KEY2"  (comma-separated)
 * or:
 *   GEMINI_API_KEY_PRIMARY="KEY1"
 *   GEMINI_API_KEY_SECONDARY="KEY2"
 * Optional:
 *   GEMINI_MODEL=gemini-2.5-flash-lite
 */

require_once __DIR__ . '/../response.php';

function gemini_model(): string {
  // Default to Gemini 2.5 Flash-Lite (low latency / cost-efficient).
  $fallback = 'gemini-2.5-flash-lite';

  // Prefer environment variable (best practice for production)
  $m = trim((string)(getenv('GEMINI_MODEL') ?: ''));

  // If not set, allow config.php / config.local.php to define a constant.
  if ($m === '' && defined('GEMINI_MODEL')) {
    $m = trim((string)constant('GEMINI_MODEL'));
  }

  return $m !== '' ? $m : $fallback;
}

function gemini_key_count(): int {
  if (defined('GEMINI_API_KEYS') && is_array(GEMINI_API_KEYS) && count(GEMINI_API_KEYS) > 0) {
    return count(GEMINI_API_KEYS);
  }

  // fallback: single key
  $k = trim((string)(getenv('GEMINI_API_KEY') ?: ''));
  if ($k === '' && defined('GEMINI_API_KEY')) {
    $k = trim((string)constant('GEMINI_API_KEY'));
  }
  return $k !== '' ? 1 : 0;
}

function gemini_key_for_attempt(int $attempt): string {
  // Prefer the rotation helper if config.php defined it.
  if (function_exists('gemini_pick_key')) {
    $k = (string)(gemini_pick_key($attempt) ?: '');
    if ($k !== '') return $k;
  }

  // Prefer GEMINI_API_KEYS if present
  if (defined('GEMINI_API_KEYS') && is_array(GEMINI_API_KEYS) && count(GEMINI_API_KEYS) > 0) {
    return (string)GEMINI_API_KEYS[$attempt % count(GEMINI_API_KEYS)];
  }

  // Fallback: single key
  $k = trim((string)(getenv('GEMINI_API_KEY') ?: ''));
  if ($k === '' && defined('GEMINI_API_KEY')) {
    $k = trim((string)constant('GEMINI_API_KEY'));
  }
  return $k;
}

function gemini_should_failover(int $status, array $data): bool {
  // Typical quota/rate errors
  if ($status === 429) return true;
  $st = strtoupper((string)($data['error']['status'] ?? ''));
  if ($st === 'RESOURCE_EXHAUSTED') return true;
  $msg = strtolower((string)($data['error']['message'] ?? ''));
  if ($msg !== '' && (str_contains($msg, 'quota') || str_contains($msg, 'rate') || str_contains($msg, 'exhaust'))) {
    return true;
  }
  return false;
}
/**
 * @return array{ok:bool, text?:string, raw?:array, error?:string, status?:int}
 */
function gemini_generate_text(string $prompt, array $opts = []): array {
  $model = $opts['model'] ?? gemini_model();
  $temperature = isset($opts['temperature']) ? (float)$opts['temperature'] : 0.4;
  $maxTokens = isset($opts['maxOutputTokens']) ? (int)$opts['maxOutputTokens'] : 800;

  $tries = max(1, gemini_key_count());
  if ($tries === 0) {
    return ['ok' => false, 'error' => 'AI is not configured on the server (missing GEMINI keys).', 'status' => 503];
  }

  $payload = [
    'contents' => [[
      'role' => 'user',
      'parts' => [
        ['text' => $prompt],
      ],
    ]],
    'generationConfig' => [
      'temperature' => $temperature,
      'maxOutputTokens' => $maxTokens,
    ],
  ];

  $lastErr = ['ok' => false, 'error' => 'AI error', 'status' => 502];

  for ($attempt = 0; $attempt < $tries; $attempt++) {
    $key = trim(gemini_key_for_attempt($attempt));
    if ($key === '') {
      $lastErr = ['ok' => false, 'error' => 'AI is not configured on the server (missing GEMINI keys).', 'status' => 503];
      continue;
    }

    $url = 'https://generativelanguage.googleapis.com/v1beta/models/' . rawurlencode((string)$model) . ':generateContent?key=' . rawurlencode($key);

    $ch = curl_init($url);
    curl_setopt_array($ch, [
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_POST => true,
      CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
      CURLOPT_POSTFIELDS => json_encode($payload, JSON_UNESCAPED_SLASHES),
      CURLOPT_CONNECTTIMEOUT => 10,
      CURLOPT_TIMEOUT => 30,
    ]);
    $raw = curl_exec($ch);
    $errno = curl_errno($ch);
    $err = curl_error($ch);
    $status = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($errno !== 0 || $raw === false) {
      $lastErr = ['ok' => false, 'error' => 'AI request failed: ' . ($err ?: 'network error'), 'status' => 502];
      // try next key on network errors
      continue;
    }

    $data = json_decode((string)$raw, true);
    if (!is_array($data)) {
      $lastErr = ['ok' => false, 'error' => 'AI returned invalid JSON.', 'status' => 502];
      continue;
    }

    if ($status >= 400) {
      $msg = (string)($data['error']['message'] ?? 'AI error');
      $lastErr = ['ok' => false, 'error' => $msg, 'status' => $status, 'raw' => $data];

      // Fail over only on quota/rate style errors
      if (gemini_should_failover($status, $data)) {
        continue;
      }
      return $lastErr;
    }

    $text = (string)($data['candidates'][0]['content']['parts'][0]['text'] ?? '');
    if (trim($text) === '') {
      $lastErr = ['ok' => false, 'error' => 'AI returned no text.', 'status' => 502, 'raw' => $data];
      continue;
    }

    return ['ok' => true, 'text' => $text, 'raw' => $data, 'status' => 200];
  }

  return $lastErr;
}
