<?php
declare(strict_types=1);

/**
 * AI abuse protection: rate limiting + daily quotas.
 *
 * We count only *real* model calls (not cache hits) by inserting into ai_request_log
 * when we actually call Gemini.
 *
 * If ai_request_log table isn't created yet, checks degrade gracefully.
 */

require_once __DIR__ . '/../response.php';

function ai_limits_for_kind(string $kind): array {
  // Tune these numbers to match your budget.
  // per_min: rolling 60 seconds window, per_day: UTC day.
  $defaults = [
    // Student-facing
    'explain_mcq'    => ['per_min' => 8,  'per_day' => 120],
    'generate_bank'  => ['per_min' => 2,  'per_day' => 12],
    'doc_chat'       => ['per_min' => 6,  'per_day' => 60],
    'theory_coach'   => ['per_min' => 4,  'per_day' => 40],

    // Admin tools
    'fill_explanations'   => ['per_min' => 2,  'per_day' => 300],
    'fill_theory_guides' => ['per_min' => 1,  'per_day' => 120],
    'quality_audit'      => ['per_min' => 1,  'per_day' => 80],
  ];

  return $defaults[$kind] ?? ['per_min' => 6, 'per_day' => 60];
}

/**
 * Enforce limits for a user and kind.
 * Returns only if allowed; otherwise sends a JSON error response (429) and exits.
 */
function ai_enforce_limits(PDO $pdo, string $userId, string $kind): void {
  $limits = ai_limits_for_kind($kind);

  try {
    // per-minute
    $stmt = $pdo->prepare(
      'SELECT COUNT(*) FROM ai_request_log
       WHERE user_id = :uid AND kind = :k AND created_at >= (UTC_TIMESTAMP() - INTERVAL 60 SECOND)'
    );
    $stmt->execute([':uid' => $userId, ':k' => $kind]);
    $minuteCount = (int)$stmt->fetchColumn();

    if ($minuteCount >= (int)$limits['per_min']) {
      json_response([
        'success' => false,
        'error' => 'Too many AI requests. Please wait a bit and try again.',
        'code' => 'AI_RATE_LIMIT',
        'retryAfterSec' => 20,
      ], 429);
    }

    // per-day (UTC day)
    $stmt = $pdo->prepare(
      'SELECT COUNT(*) FROM ai_request_log
       WHERE user_id = :uid AND kind = :k AND created_at >= UTC_DATE()'
    );
    $stmt->execute([':uid' => $userId, ':k' => $kind]);
    $dayCount = (int)$stmt->fetchColumn();

    if ($dayCount >= (int)$limits['per_day']) {
      json_response([
        'success' => false,
        'error' => 'Daily AI quota reached. Try again tomorrow.',
        'code' => 'AI_DAILY_QUOTA',
      ], 429);
    }
  } catch (Throwable $e) {
    // If table doesn't exist yet or DB error, don't block the user.
  }
}

/**
 * Log an AI model call for quota/rate tracking.
 */
function ai_log_call(PDO $pdo, string $userId, string $kind, string $promptHash): void {
  try {
    $stmt = $pdo->prepare(
      'INSERT INTO ai_request_log (user_id, kind, prompt_hash, created_at)
       VALUES (:uid, :k, :h, UTC_TIMESTAMP())'
    );
    $stmt->execute([':uid' => $userId, ':k' => $kind, ':h' => $promptHash]);
  } catch (Throwable $e) {
    // ignore
  }
}

/**
 * Extract a JSON object/array from a response that may contain extra text.
 */
function ai_extract_json(string $text): ?string {
  $t = trim($text);

  // Strip code fences
  $t = preg_replace('/^```(json)?\s*/i', '', $t) ?? $t;
  $t = preg_replace('/\s*```$/', '', $t) ?? $t;
  $t = trim($t);

  if ($t === '') return null;

  // If already valid JSON, return as-is
  $tmp = json_decode($t, true);
  if (is_array($tmp)) return $t;

  // Try to locate the first {...} or [...]
  $startObj = strpos($t, '{');
  $startArr = strpos($t, '[');
  if ($startObj === false && $startArr === false) return null;

  $start = $startObj;
  if ($start === false || ($startArr !== false && $startArr < $startObj)) $start = $startArr;

  // naive scan for matching end by taking last } or ]
  $endObj = strrpos($t, '}');
  $endArr = strrpos($t, ']');
  $end = $endObj;
  if ($end === false || ($endArr !== false and $endArr > $endObj)) $end = $endArr;
  if ($end === false or $end <= $start) return null;

  $slice = substr($t, $start, $end - $start + 1);
  $tmp2 = json_decode($slice, true);
  if (is_array($tmp2)) return $slice;

  return null;
}
