<?php
declare(strict_types=1);

/**
 * Lightweight retrieval helpers for Doc Chat (no embeddings).
 *
 * Goal: pick the most relevant parts of a PDF's extracted text so we don't
 * send a giant context window to the model (cheaper + more accurate).
 */

/**
 * Split pdftotext output into pages.
 * pdftotext commonly uses form-feed (\f) between pages.
 *
 * @return array<int, string> pages (1-indexed meaning is handled by caller)
 */
function rag_split_pages(string $text): array {
  $t = (string)$text;
  $t = str_replace(["\r\n", "\r"], "\n", $t);

  // Split on one or more form feeds.
  $pages = preg_split('/\f+/', $t);
  if (!is_array($pages) || count($pages) === 0) {
    return [trim($t)];
  }

  $out = [];
  foreach ($pages as $p) {
    $p = trim((string)$p);
    if ($p === '') continue;
    $out[] = $p;
  }

  return count($out) ? $out : [trim($t)];
}

/**
 * Very small stopword list (kept as a set for O(1) lookup).
 *
 * @return array<string, true>
 */
function rag_stopwords(): array {
  static $set = null;
  if (is_array($set)) return $set;

  $words = [
    'a','an','and','are','as','at','be','because','been','before','being','below','between',
    'both','but','by','can','could','did','do','does','doing','down','during','each','few',
    'for','from','further','had','has','have','having','he','her','here','hers','herself',
    'him','himself','his','how','i','if','in','into','is','it','its','itself','just','me',
    'more','most','my','myself','no','nor','not','now','of','off','on','once','only','or',
    'other','our','ours','ourselves','out','over','own','same','she','should','so','some',
    'such','than','that','the','their','theirs','them','themselves','then','there','these',
    'they','this','those','through','to','too','under','until','up','very','was','we','were',
    'what','when','where','which','while','who','whom','why','with','would','you','your','yours',
    'yourself','yourselves',
  ];

  $set = [];
  foreach ($words as $w) $set[$w] = true;
  return $set;
}

/**
 * Tokenize text into a unique list of meaningful tokens.
 *
 * @return string[]
 */
function rag_tokens(string $text): array {
  $t = mb_strtolower((string)$text);
  // Keep letters + numbers, replace everything else with spaces.
  $t = preg_replace('/[^\p{L}\p{N}]+/u', ' ', $t) ?? $t;
  $parts = preg_split('/\s+/', trim($t)) ?: [];

  $stop = rag_stopwords();
  $out = [];
  foreach ($parts as $p) {
    $p = (string)$p;
    if ($p === '') continue;
    if (isset($stop[$p])) continue;
    // Drop very short tokens except pure numbers.
    $isNum = preg_match('/^\d+$/', $p) === 1;
    if (!$isNum && mb_strlen($p) < 3) continue;
    $out[$p] = true;
  }

  return array_keys($out);
}

/**
 * Chunk a page into ~chunkChars chunks, with a small overlap.
 *
 * @return string[]
 */
function rag_chunk_page(string $pageText, int $chunkChars = 1400, int $overlapChars = 180): array {
  $t = str_replace(["\r\n", "\r"], "\n", (string)$pageText);
  $paras = preg_split("/\n\s*\n/", $t);
  if (!is_array($paras) || count($paras) === 0) {
    $paras = [$t];
  }

  $chunks = [];
  $buf = '';

  $flush = function () use (&$chunks, &$buf) {
    $b = trim((string)$buf);
    if ($b !== '') $chunks[] = $b;
    $buf = '';
  };

  foreach ($paras as $para) {
    $p = trim((string)$para);
    if ($p === '') continue;

    // If this single paragraph is huge, split it by characters.
    while (mb_strlen($p) > $chunkChars) {
      $seg = trim(mb_substr($p, 0, $chunkChars));
      if ($seg !== '') {
        if (trim($buf) !== '') $flush();
        $chunks[] = $seg;
      }
      $p = mb_substr($p, max(0, $chunkChars - $overlapChars));
      $p = trim((string)$p);
      if ($p === '') break;
    }

    if ($p === '') continue;

    if ($buf === '') {
      $buf = $p;
      continue;
    }

    // Add paragraph; if it overflows, flush and start new chunk with overlap.
    $candidate = $buf . "\n\n" . $p;
    if (mb_strlen($candidate) <= $chunkChars) {
      $buf = $candidate;
      continue;
    }

    // Flush current buffer.
    $prev = $buf;
    $flush();

    // Start a new chunk with a tail overlap.
    $tail = '';
    if ($overlapChars > 0) {
      $len = mb_strlen($prev);
      $start = max(0, $len - $overlapChars);
      $tail = trim(mb_substr($prev, $start));
    }
    $buf = ($tail !== '' ? ($tail . "\n\n") : '') . $p;
  }

  if (trim($buf) !== '') $flush();
  return $chunks;
}

/**
 * Build chunk objects from extracted text.
 *
 * @return array<int, array{page:int, chunkIndex:int, text:string}>
 */
function rag_build_chunks(string $extractedText, int $chunkChars = 1400, int $overlapChars = 180): array {
  $pages = rag_split_pages($extractedText);
  $out = [];

  $pageNo = 1;
  foreach ($pages as $p) {
    $chunks = rag_chunk_page($p, $chunkChars, $overlapChars);
    $ci = 1;
    foreach ($chunks as $c) {
      $out[] = ['page' => $pageNo, 'chunkIndex' => $ci, 'text' => $c];
      $ci++;
    }
    $pageNo++;
  }

  return $out;
}

/**
 * Rank chunks by simple token overlap with the question.
 *
 * @param array<int, array{page:int, chunkIndex:int, text:string}> $chunks
 * @return array<int, array{page:int, chunkIndex:int, text:string, score:float}>
 */
function rag_rank_chunks(string $question, array $chunks): array {
  $qTokens = rag_tokens($question);
  if (count($qTokens) === 0) {
    // If tokenization fails (e.g. only stopwords), give everything a tiny score.
    return array_map(static fn($c) => $c + ['score' => 0.001], $chunks);
  }

  $ranked = [];
  foreach ($chunks as $c) {
    $txt = (string)($c['text'] ?? '');
    $cTokens = rag_tokens($txt);
    if (count($cTokens) === 0) {
      $ranked[] = $c + ['score' => 0.0];
      continue;
    }

    $set = [];
    foreach ($cTokens as $t) $set[$t] = true;

    $hits = 0.0;
    foreach ($qTokens as $qt) {
      if (!isset($set[$qt])) continue;
      // Slightly reward longer, more specific tokens.
      $len = mb_strlen($qt);
      $hits += ($len >= 8 ? 2.0 : ($len >= 5 ? 1.5 : 1.0));
    }

    // Normalize a bit by chunk token count (prevents super-long chunks winning).
    $score = $hits / sqrt(max(1.0, (float)count($cTokens)));
    $ranked[] = $c + ['score' => $score];
  }

  usort($ranked, static function ($a, $b) {
    return ($b['score'] <=> $a['score']);
  });

  return $ranked;
}

/**
 * Select the best chunks for answering.
 *
 * @param array<int, array{page:int, chunkIndex:int, text:string}> $chunks
 * @return array<int, array{page:int, chunkIndex:int, text:string, score:float}>
 */
function rag_select_chunks(string $question, array $chunks, int $topK = 6, float $minScore = 0.05): array {
  $ranked = rag_rank_chunks($question, $chunks);
  if (count($ranked) === 0) return [];

  $picked = [];
  foreach ($ranked as $c) {
    if (count($picked) >= $topK) break;
    if ((float)($c['score'] ?? 0.0) < $minScore) continue;
    $picked[] = $c;
  }

  // If nothing passes the threshold, still send the top chunk (gives the model something).
  if (count($picked) === 0) {
    $picked[] = $ranked[0];
  }

  // Sort picked chunks by page/position so reading order is natural.
  usort($picked, static function ($a, $b) {
    $pa = (int)$a['page'];
    $pb = (int)$b['page'];
    if ($pa !== $pb) return $pa <=> $pb;
    return ((int)$a['chunkIndex']) <=> ((int)$b['chunkIndex']);
  });

  return $picked;
}

// ---------------------------------------------------------------------
// Compatibility wrappers
// ---------------------------------------------------------------------
// Some endpoints in this repo call these helper names.
// Keep them as thin wrappers around the newer functions above.

/**
 * Build chunks with global chunkIndex + lightweight preview.
 *
 * @return array<int, array{chunkIndex:int,page:int,pageChunkIndex:int,text:string,preview:string}>
 */
function rag_make_chunks(string $extractedText, int $chunkChars = 1400, int $overlapChars = 180): array {
  $built = rag_build_chunks($extractedText, $chunkChars, $overlapChars);

  $out = [];
  $i = 1;
  foreach ($built as $c) {
    $txt = trim((string)($c['text'] ?? ''));
    if ($txt === '') continue;

    $prev = preg_replace('/\s+/', ' ', $txt) ?? $txt;
    $prev = trim((string)$prev);
    if (mb_strlen($prev) > 180) $prev = mb_substr($prev, 0, 180) . '…';

    $out[] = [
      'chunkIndex' => $i,
      'page' => (int)($c['page'] ?? 1),
      'pageChunkIndex' => (int)($c['chunkIndex'] ?? 1),
      'text' => $txt,
      'preview' => $prev,
    ];
    $i++;
  }

  return $out;
}

/**
 * Pick top chunks for a question.
 *
 * @param array<int, array{chunkIndex:int,page?:int,text:string,preview?:string}> $chunks
 * @return array<int, array{chunkIndex:int,text:string,score:int,preview:string,page?:int,pageChunkIndex?:int}>
 */
function rag_select_top_chunks(array $chunks, string $question, int $topK = 6): array {
  $qTokens = rag_tokens($question);
  $ranked = [];

  foreach ($chunks as $c) {
    $txt = (string)($c['text'] ?? '');
    $cTokens = rag_tokens($txt);

    $hits = 0.0;
    if (count($qTokens) > 0 && count($cTokens) > 0) {
      $set = [];
      foreach ($cTokens as $t) $set[$t] = true;
      foreach ($qTokens as $qt) {
        if (!isset($set[$qt])) continue;
        $len = mb_strlen($qt);
        $hits += ($len >= 8 ? 2.0 : ($len >= 5 ? 1.5 : 1.0));
      }
    }

    $norm = sqrt(max(1.0, (float)count($cTokens)));
    $scoreFloat = $hits / $norm;
    $score = (int)round($scoreFloat * 100); // stable integer for UI

    $ranked[] = $c + ['score' => $score];
  }

  usort($ranked, static function ($a, $b) {
    return ((int)($b['score'] ?? 0)) <=> ((int)($a['score'] ?? 0));
  });

  $picked = [];
  foreach ($ranked as $c) {
    if (count($picked) >= $topK) break;
    if ((int)($c['score'] ?? 0) <= 0) continue;
    $picked[] = $c;
  }

  if (count($picked) === 0 && count($ranked) > 0) {
    $picked[] = $ranked[0];
  }

  // ensure preview exists
  $out = [];
  foreach ($picked as $c) {
    $prev = (string)($c['preview'] ?? '');
    if ($prev === '') {
      $t = preg_replace('/\s+/', ' ', (string)($c['text'] ?? '')) ?? '';
      $t = trim((string)$t);
      if (mb_strlen($t) > 180) $t = mb_substr($t, 0, 180) . '…';
      $prev = $t;
    }

    $out[] = [
      'chunkIndex' => (int)($c['chunkIndex'] ?? 0),
      'page' => isset($c['page']) ? (int)$c['page'] : null,
      'pageChunkIndex' => isset($c['pageChunkIndex']) ? (int)$c['pageChunkIndex'] : null,
      'text' => (string)($c['text'] ?? ''),
      'score' => (int)($c['score'] ?? 0),
      'preview' => $prev,
    ];
  }

  return $out;
}
