<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

require_once __DIR__ . '/_gemini.php';
require_once __DIR__ . '/_limits.php';
require_once __DIR__ . '/_extract.php';
require_once __DIR__ . '/_cache.php';
require_once __DIR__ . '/_rag.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$body = read_json();
require_fields($body, ['docType', 'docId', 'question']);

$docType = strtolower(trim((string)$body['docType']));
$docId = trim((string)$body['docId']);
$question = trim((string)$body['question']);

if ($docId === '' || $question === '') {
  json_response(['success' => false, 'error' => 'Missing docId/question'], 422);
}

$tbl = null;
switch ($docType) {
  case 'materials':
    $tbl = 'materials';
    break;
  case 'pastquestions':
  case 'past_questions':
    $tbl = 'past_questions';
    break;
  default:
    json_response(['success' => false, 'error' => 'Invalid docType'], 422);
}

$row = null;
try {
  $stmt = $pdo->prepare("SELECT id, course_id AS courseId, title, file_path AS filePath" . ($tbl === 'materials' ? ", type" : "") . ", created_by AS createdBy" . " FROM {$tbl} WHERE id = :id LIMIT 1");
  $stmt->execute([':id' => $docId]);
  $row = $stmt->fetch() ?: null;
} catch (Throwable $e) {
  json_response(['success' => false, 'error' => 'Document not found'], 404);
}

if (!$row) json_response(['success' => false, 'error' => 'Document not found'], 404);

$courseId = (string)$row['courseId'];

// Ensure access to this course
$chk = $pdo->prepare('SELECT 1 FROM user_courses WHERE user_id = :uid AND course_id = :cid LIMIT 1');
$chk->execute([':uid' => $me['id'], ':cid' => $courseId]);
if (!(bool)$chk->fetchColumn()) {
  json_response(['success' => false, 'error' => 'You do not have access to this course'], 403);
}

// Fetch extracted text if column exists
$extracted = null;
try {
  $stmt = $pdo->prepare("SELECT extracted_text AS extractedText FROM {$tbl} WHERE id = :id LIMIT 1");
  $stmt->execute([':id' => $docId]);
  $extracted = $stmt->fetchColumn();
  if (!is_string($extracted) || trim($extracted) === '') $extracted = null;
} catch (Throwable $e) {
  $extracted = null; // column may not exist yet
}

// If no extracted text, attempt best-effort extraction for PDFs
if ($extracted === null) {
  $apiRoot = realpath(__DIR__ . '/..');
  $abs = $apiRoot ? resolve_api_file_path($apiRoot, (string)$row['filePath']) : null;

  if ($abs) {
    $maybeOut = pdf_extract_text_with_reason($abs);
    $maybe = $maybeOut['text'] ?? null;
    $extractReason = (string)($maybeOut['reason'] ?? '');
    if (is_string($maybe) && trim($maybe) !== '') {
      $extracted = (string)$maybe;

      // Save back if columns exist
      try {
        $u = $pdo->prepare("UPDATE {$tbl} SET extracted_text = :t, extracted_at = UTC_TIMESTAMP() WHERE id = :id");
        $u->execute([':t' => $extracted, ':id' => $docId]);
      } catch (Throwable $e) {
        // ignore
      }
    } else {
      // Keep a lightweight hint for the client
      $extracted = null;
      $GLOBALS['__DOC_CHAT_EXTRACT_REASON__'] = $extractReason;
    }
  }
}

if ($extracted === null) {
  $reason = (string)($GLOBALS['__DOC_CHAT_EXTRACT_REASON__'] ?? '');
  $code = 'NO_EXTRACTED_TEXT';
  $msg = 'This PDF has no readable text (likely scanned). Re-upload a text-based PDF or add a text summary.';

  if (in_array($reason, ['SHELL_EXEC_DISABLED', 'PDFTOTEXT_MISSING'], true)) {
    $code = 'TEXT_EXTRACTION_UNAVAILABLE';
    $msg = 'Text extraction is not available on this server. Install pdftotext (Poppler) or enable shell_exec, then try again. As a workaround, upload a text-based PDF or add a text summary.';
  } else if ($reason === 'FILE_MISSING') {
    $code = 'FILE_NOT_FOUND';
    $msg = 'The file could not be found on the server. Please re-upload the document.';
  }

  json_response([
    'success' => false,
    'error' => $msg,
    'code' => $code,
    'reason' => $reason,
  ], 409);
}

// ----------------------------
// RAG-style chunk retrieval (cheap; no embeddings)
// ----------------------------
$chunkChars = 1400;
$overlapChars = 200;
$topK = 6;
$algoVersion = 2;

$exHash = hash('sha256', $extracted);
$cacheKey = ai_cache_key('doc_chat', [
  'docType' => $docType,
  'docId' => $docId,
  'q' => mb_strtolower(preg_replace('/\s+/', ' ', $question) ?? $question),
  'exHash' => $exHash,
  'model' => gemini_model(),
  'v' => $algoVersion,
  'chunkChars' => $chunkChars,
  'overlapChars' => $overlapChars,
  'topK' => $topK,
]);

$cachedJson = ai_cache_get($pdo, $cacheKey);
if (is_string($cachedJson)) {
  $tmp = json_decode($cachedJson, true);
  if (is_array($tmp) && isset($tmp['answer']) && is_string($tmp['answer'])) {
    json_response([
      'success' => true,
      'data' => [
        'answer' => (string)$tmp['answer'],
        'model' => (string)($tmp['model'] ?? gemini_model()),
        'sources' => is_array($tmp['sources'] ?? null) ? $tmp['sources'] : [],
        'cached' => true,
      ],
    ]);
  }
}

// Rate limit + daily quota (counts only real model calls)
ai_enforce_limits($pdo, (string)$me['id'], 'doc_chat');
$callHash = hash('sha256', json_encode([
  'docType' => $docType,
  'docId' => $docId,
  'q' => $question,
  'exHash' => $exHash,
  'v' => $algoVersion,
], JSON_UNESCAPED_SLASHES));
ai_log_call($pdo, (string)$me['id'], 'doc_chat', $callHash);

$chunks = rag_make_chunks($extracted, $chunkChars, $overlapChars);
if (count($chunks) === 0) {
  json_response([
    'success' => false,
    'error' => 'Could not chunk extracted text.',
  ], 500);
}

$top = rag_select_top_chunks($chunks, $question, $topK);
if (count($top) === 0) {
  // Fallback: use the very first chunks
  $top = rag_select_top_chunks($chunks, 'content', $topK);
}

// Build prompt excerpts
$excerpts = "";
$sources = [];
foreach ($top as $item) {
  $idx = (int)($item['chunkIndex'] ?? 0);
  $txt = (string)($item['text'] ?? '');
  $score = (int)($item['score'] ?? 0);
  $preview = (string)($item['preview'] ?? '');
  $page = isset($item['page']) ? (int)$item['page'] : null;
  $pageChunkIndex = isset($item['pageChunkIndex']) ? (int)$item['pageChunkIndex'] : null;

  $excerpts .= "\n[EXCERPT {$idx}]" . ($page ? " (page {$page})" : "") . "\n{$txt}\n";
  $sources[] = [
    'chunk' => $idx,
    'page' => $page,
    'pageChunk' => $pageChunkIndex,
    'score' => $score,
    'preview' => $preview,
  ];
}

$title = (string)($row['title'] ?? '');
$courseLabel = $courseId;

$prompt = "You are a helpful study assistant for Nigerian university students.\n"
  . "Safety rules: do not include hate/harassment/sexual content, do not provide illegal instructions, and do not request personal data.\n"
  . "Answer the user's question using ONLY the EXCERPTS below.\n"
  . "If the answer is not in the excerpts, say: Not found in this material. Then suggest 1-2 keywords the user can search for.\n"
  . "When you use information from an excerpt, add a citation like [EXCERPT 3].\n"
  . "Keep the answer clear and concise (5-12 lines).\n\n"
  . "Material title: {$title}\n"
  . "Course: {$courseLabel}\n\n"
  . "EXCERPTS:\n" . $excerpts . "\n"
  . "USER QUESTION:\n" . $question;

$resp = gemini_generate_text($prompt, [
  'temperature' => 0.3,
  'maxOutputTokens' => 900,
]);

if (!$resp['ok']) {
  json_response(['success' => false, 'error' => $resp['error'] ?? 'AI failed'], (int)($resp['status'] ?? 502));
}

$answer = trim((string)$resp['text']);

// Cache answer + sources (7 days)
ai_cache_set(
  $pdo,
  $cacheKey,
  'doc_chat',
  gemini_model(),
  json_encode(['answer' => $answer, 'model' => gemini_model(), 'sources' => $sources], JSON_UNESCAPED_SLASHES),
  60 * 60 * 24 * 7
);

json_response([
  'success' => true,
  'data' => [
    'answer' => $answer,
    'model' => gemini_model(),
    'sources' => $sources,
    'cached' => false,
  ],
]);
