<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/_gemini.php';
require_once __DIR__ . '/_limits.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$body = read_json();
require_fields($body, ['bankId', 'questionId', 'mode']);

$bankId = trim((string)$body['bankId']);
$questionId = trim((string)$body['questionId']);
$mode = trim((string)$body['mode']); // hint | full

$selectedIndex = array_key_exists('selectedIndex', $body) ? (int)$body['selectedIndex'] : null;

if (!in_array($mode, ['hint', 'full'], true)) {
  json_response(['success' => false, 'error' => 'Invalid mode'], 422);
}

function option_letter(int $i): string {
  return chr(65 + $i);
}

/**
 * Remove code fences and try to parse JSON robustly.
 * Returns array if decode works, otherwise null.
 */
function extract_json_array(string $text): ?array {
  $text = trim($text);

  // Strip common ```json fences
  $text = preg_replace('/^```(json)?\s*/i', '', $text) ?? $text;
  $text = preg_replace('/\s*```$/', '', $text) ?? $text;
  $text = trim($text);

  // 1) Direct decode
  $decoded = json_decode($text, true);
  if (is_array($decoded)) return $decoded;

  // 2) Try to find the first JSON object block in the text
  // This is a best-effort fallback for models that add extra text around JSON.
  if (preg_match('/\{[\s\S]*\}/', $text, $m)) {
    $candidate = trim((string)$m[0]);
    $decoded2 = json_decode($candidate, true);
    if (is_array($decoded2)) return $decoded2;
  }

  return null;
}

/**
 * Light normalisation so your UI stays stable even if the model returns slightly off.
 */
function normalise_full_result(array $result, array $options): array {
  // Ensure required keys exist
  $result += [
    'isCorrect' => null,
    'correctOption' => null,
    'explanation' => '',
    'steps' => [],
    'whyOthersAreWrong' => [],
    'keyTakeaway' => '',
  ];

  // Coerce types
  if (!is_bool($result['isCorrect']) && $result['isCorrect'] !== null) $result['isCorrect'] = null;

  if (!is_string($result['correctOption']) && $result['correctOption'] !== null) {
    $result['correctOption'] = null;
  }

  if (!is_string($result['explanation'])) $result['explanation'] = (string)$result['explanation'];
  if (!is_string($result['keyTakeaway'])) $result['keyTakeaway'] = (string)$result['keyTakeaway'];

  if (!is_array($result['steps'])) $result['steps'] = [];
  if (!is_array($result['whyOthersAreWrong'])) $result['whyOthersAreWrong'] = [];

  // Trim items
  $result['steps'] = array_values(array_filter(array_map(function ($x) {
    $s = trim((string)$x);
    return $s === '' ? null : $s;
  }, $result['steps'])));

  $result['whyOthersAreWrong'] = array_values(array_filter(array_map(function ($x) {
    $s = trim((string)$x);
    return $s === '' ? null : $s;
  }, $result['whyOthersAreWrong'])));

  // If correctOption looks like an index, try to convert (rare but happens)
  // Example: "1" => "B"
  if (is_string($result['correctOption']) && preg_match('/^\d+$/', $result['correctOption'])) {
    $idx = (int)$result['correctOption'];
    if ($idx >= 0 && $idx < count($options)) $result['correctOption'] = option_letter($idx);
  }

  // Uppercase correctOption if present
  if (is_string($result['correctOption'])) {
    $result['correctOption'] = strtoupper(trim($result['correctOption']));
  }

  return $result;
}

// Load question from DB (never trust client content)
$q = $pdo->prepare(
  'SELECT q.prompt, q.`options` AS options, q.answer_index AS answerIndex, q.explanation AS explanation
   FROM questions q
   WHERE q.bank_id = :bid AND q.id = :qid
   LIMIT 1'
);
$q->execute([':bid' => $bankId, ':qid' => $questionId]);
$row = $q->fetch();

if (!$row) {
  json_response(['success' => false, 'error' => 'Question not found'], 404);
}

$prompt = (string)($row['prompt'] ?? '');

$options = json_decode((string)($row['options'] ?? '[]'), true);
if (!is_array($options)) $options = [];

$options = array_values(array_map(function ($x) {
  return (string)$x;
}, $options));

$answerIndex = (int)($row['answerIndex'] ?? 0);
if ($answerIndex < 0) $answerIndex = 0;
if ($answerIndex >= count($options) && count($options) > 0) $answerIndex = 0;

// Validate selectedIndex bounds (if provided)
if ($selectedIndex !== null) {
  if ($selectedIndex < 0 || $selectedIndex >= count($options)) {
    $selectedIndex = null;
  }
}

// Cache key based on immutable content
$promptHash = hash('sha256', json_encode([
  'mode' => $mode,
  'prompt' => $prompt,
  'options' => $options,
  'answerIndex' => $answerIndex,
], JSON_UNESCAPED_SLASHES));

// Try cache if table exists
$cached = null;
try {
  $c = $pdo->prepare(
    'SELECT response_text
     FROM ai_mcq_explanations
     WHERE question_id = :qid AND mode = :mode AND prompt_hash = :ph
     LIMIT 1'
  );
  $c->execute([':qid' => $questionId, ':mode' => $mode, ':ph' => $promptHash]);
  $cached = $c->fetchColumn();
} catch (Throwable $e) {
  // table may not exist yet; ignore
}

if (is_string($cached) && trim($cached) !== '') {
  $decoded = json_decode($cached, true);

  // If cache holds non-JSON, fall back to text wrapper
  $result = is_array($decoded) ? $decoded : ['text' => $cached];

  // Optional: normalise for full mode
  if ($mode === 'full' && is_array($decoded)) {
    $result = normalise_full_result($decoded, $options);
  }

  json_response([
    'success' => true,
    'data' => [
      'mode' => $mode,
      'cached' => true,
      'result' => $result,
    ],
  ]);
}

// Rate limit + daily quota (counts only real model calls)
ai_enforce_limits($pdo, (string)$me['id'], 'explain_mcq');
$callHash = hash('sha256', $promptHash . '|' . (string)$selectedIndex);
ai_log_call($pdo, (string)$me['id'], 'explain_mcq', $callHash);

// Build options text
$optsText = '';
foreach ($options as $i => $opt) {
  $optsText .= option_letter((int)$i) . '. ' . $opt . "\n";
}

// Build tutor prompt
if ($mode === 'hint') {
  $aiPrompt =
    "You are an exam coach tutoring a Nigerian university student.\n"
    . "Your job: give ONE short hint (1 to 2 sentences) that helps the student solve the MCQ.\n"
    . "Rules:\n"
    . "1) Do NOT reveal the correct option letter or index.\n"
    . "2) Do NOT quote the correct option text verbatim.\n"
    . "3) Do NOT say or imply which option is correct.\n"
    . "4) Focus on the concept, definition, clue words to look for, and a quick elimination tip.\n"
    . "Style: simple, direct, British English.\n"
    . "Output: Return ONLY valid JSON, no markdown, no extra keys.\n"
    . "JSON shape exactly: {\"hint\":\"...\"}\n\n"
    . "Question:\n" . $prompt . "\n\n"
    . "Options:\n" . $optsText;
} else {
  $sel = ($selectedIndex === null) ? null : ($options[$selectedIndex] ?? null);
  $isCorrect = ($selectedIndex === null) ? null : ((int)$selectedIndex === (int)$answerIndex);

  $aiPrompt =
    "You are an exam coach tutoring a Nigerian university student.\n"
    . "Task: explain the correct answer to this MCQ and give feedback on the student selection.\n"
    . "Use British English. Keep it concise and exam-focused.\n\n"
    . "Hard rules:\n"
    . "1) Return ONLY valid JSON. No markdown. No code fences. No extra text.\n"
    . "2) Do not invent facts outside the question and options.\n"
    . "3) If the question is ambiguous, say so briefly, then still explain using the provided correct option.\n"
    . "4) Do not repeat the full option texts. Refer to options by letter only.\n"
    . "5) Keep each array item to one sentence.\n\n"
    . "JSON schema (must match exactly):\n"
    . "{\n"
    . "  \"isCorrect\": true|false|null,\n"
    . "  \"correctOption\": \"A\"|\"B\"|\"C\"|\"D\"|\"E\"|null,\n"
    . "  \"explanation\": \"2 to 4 sentences\",\n"
    . "  \"steps\": [\"3 to 5 short steps\"],\n"
    . "  \"whyOthersAreWrong\": [\"One short reason for each wrong option in the form 'A: ...'\"],\n"
    . "  \"keyTakeaway\": \"One short sentence\"\n"
    . "}\n\n"
    . "Content rules:\n"
    . "- Start explanation with: \"Your choice (X) is correct/incorrect.\" If student selected is null, start with: \"Here is how to get the correct answer.\".\n"
    . "- In steps, teach a reusable rule or definition.\n"
    . "- In whyOthersAreWrong, include ALL incorrect letters that exist in the options.\n"
    . "- No praise beyond the first sentence. No fluff.\n\n"
    . "Question:\n" . $prompt . "\n\n"
    . "Options:\n" . $optsText . "\n"
    . "Correct option letter: " . option_letter($answerIndex) . "\n"
    . "Student selected: " . ($selectedIndex === null ? "null" : option_letter((int)$selectedIndex)) . "\n"
    . "Student selected text: " . ($sel === null ? "null" : (string)$sel) . "\n"
    . "isCorrect: " . ($isCorrect === null ? "null" : ($isCorrect ? "true" : "false")) . "\n";
}

$resp = gemini_generate_text($aiPrompt, [
  'temperature' => $mode === 'hint' ? 0.3 : 0.4,
  'maxOutputTokens' => $mode === 'hint' ? 300 : 900,
]);

if (!$resp['ok']) {
  json_response([
    'success' => false,
    'error' => $resp['error'] ?? 'AI failed',
  ], (int)($resp['status'] ?? 502));
}

$text = trim((string)$resp['text']);

// Parse JSON robustly
$decoded = extract_json_array($text);

if ($decoded === null) {
  // Fallback if the model still refuses to give JSON
  $result = ['text' => $text];
} else {
  // Normalise based on mode
  if ($mode === 'hint') {
    $decoded += ['hint' => ''];
    if (!is_string($decoded['hint'])) $decoded['hint'] = (string)$decoded['hint'];
    $result = ['hint' => trim($decoded['hint'])];
  } else {
    $result = normalise_full_result($decoded, $options);
  }
}

// Save cache if table exists
try {
  $ins = $pdo->prepare(
    'INSERT INTO ai_mcq_explanations (bank_id, question_id, mode, model, prompt_hash, response_text, created_at)
     VALUES (:bid, :qid, :mode, :model, :ph, :rt, UTC_TIMESTAMP())'
  );
  $ins->execute([
    ':bid' => $bankId,
    ':qid' => $questionId,
    ':mode' => $mode,
    ':model' => gemini_model(),
    ':ph' => $promptHash,
    ':rt' => json_encode($result, JSON_UNESCAPED_SLASHES),
  ]);
} catch (Throwable $e) {
  // ignore cache failures
}

json_response([
  'success' => true,
  'data' => [
    'mode' => $mode,
    'cached' => false,
    'result' => $result,
  ],
]);
