<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/_gemini.php';
require_once __DIR__ . '/_limits.php';
require_once __DIR__ . '/_cache.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$body = read_json();
require_fields($body, ['courseId']);

$courseId = trim((string)$body['courseId']);
$topic = trim((string)($body['topic'] ?? ''));
$difficulty = trim((string)($body['difficulty'] ?? 'mixed')); // easy|medium|hard|mixed
$count = (int)($body['count'] ?? 8);

if ($count < 3) $count = 3;
if ($count > 20) $count = 20;

if (!in_array($difficulty, ['easy', 'medium', 'hard', 'mixed'], true)) {
  $difficulty = 'mixed';
}

// Ensure the user has access to this course
$chk = $pdo->prepare('SELECT 1 FROM user_courses WHERE user_id = :uid AND course_id = :cid LIMIT 1');
$chk->execute([':uid' => $me['id'], ':cid' => $courseId]);
if (!(bool)$chk->fetchColumn()) {
  json_response(['success' => false, 'error' => 'You do not have access to this course'], 403);
}

// Course metadata (for better prompts)
$cstmt = $pdo->prepare('SELECT code, title, level FROM courses WHERE id = :id LIMIT 1');
$cstmt->execute([':id' => $courseId]);
$course = $cstmt->fetch() ?: null;
$courseCode = $course ? (string)$course['code'] : $courseId;
$courseTitle = $course ? (string)$course['title'] : '';

$bankId = 'ai_' . substr(hash('sha1', uuidv4() . '|' . microtime(true)), 0, 12);
$bankTitle = 'AI Quiz: ' . $courseCode;
if ($topic !== '') $bankTitle .= ' — ' . $topic;

// --------------------------------------------------
// Cache: reuse same generated quiz (per day bucket) to save API calls.
// If the user wants a fresh set, they can slightly change topic/difficulty/count.
// --------------------------------------------------
$cachedHit = false;
$cacheBucket = gmdate('Y-m-d'); // daily bucket keeps quizzes stable for a day
$cacheKey = ai_cache_key('generate_bank', [
  'bucket' => $cacheBucket,
  'courseId' => $courseId,
  'topic' => $topic,
  'difficulty' => $difficulty,
  'count' => $count,
  'model' => gemini_model(),
  'temperature' => 0.6,
  'maxOutputTokens' => 1400,
]);

$cachedJson = ai_cache_get($pdo, $cacheKey);
if (is_string($cachedJson)) {
  $tmp = json_decode($cachedJson, true);
  if (is_array($tmp) && isset($tmp['questions']) && is_array($tmp['questions'])) {
    $clean = $tmp['questions'];
    $cachedHit = is_array($clean) && count($clean) >= 3;
  }
}

// Rate limit + daily quota (counts only real model calls)
if (!$cachedHit) {
  ai_enforce_limits($pdo, (string)$me['id'], 'generate_bank');
}

$prompt = "You are generating a multiple-choice quiz for a Nigerian university student.\n"
  . "Course: {$courseCode}" . ($courseTitle !== '' ? " ({$courseTitle})" : '') . "\n"
  . "Difficulty: {$difficulty}\n"
  . ($topic !== '' ? "Topic focus: {$topic}\n" : "")
  . "Generate EXACTLY {$count} questions. Each question must have 4 options.\n"
  . "Return ONLY valid JSON with this exact shape (no markdown):\n"
  . "{\n  \"questions\": [\n    {\n      \"prompt\": \"...\",\n      \"options\": [\"A\",\"B\",\"C\",\"D\"],\n      \"answerIndex\": 0,\n      \"explanation\": \"...\"\n    }\n  ]\n}\n\n"
  . "Rules:\n"
  . "- Keep prompts clear and not too long.\n"
  . "- Ensure exactly one correct answer.\n"
  . "- Explanations must be 1-3 sentences.\n"
  . "- Avoid controversial political/medical claims; stay academic.\n";


if (!$cachedHit) {
  $callHash = hash('sha256', json_encode(['courseId' => $courseId, 'topic' => $topic, 'difficulty' => $difficulty, 'count' => $count, 'bucket' => $cacheBucket], JSON_UNESCAPED_SLASHES));
  ai_log_call($pdo, (string)$me['id'], 'generate_bank', $callHash);

  $resp = gemini_generate_text($prompt, [
    'temperature' => 0.6,
    'maxOutputTokens' => 1400,
  ]);

  if (!$resp['ok']) {
    json_response([
      'success' => false,
      'error' => $resp['error'] ?? 'AI failed',
    ], (int)($resp['status'] ?? 502));
  }

  $text = trim((string)$resp['text']);
  $text = preg_replace('/^```(json)?\s*/i', '', $text) ?? $text;
  $text = preg_replace('/\s*```$/', '', $text) ?? $text;
  $text = trim($text);

  $data = json_decode($text, true);
  if (!is_array($data) || !isset($data['questions']) || !is_array($data['questions'])) {
    json_response(['success' => false, 'error' => 'AI returned invalid quiz format'], 502);
  }

  $questions = $data['questions'];
  // clamp in case model returns extra
  $questions = array_slice($questions, 0, $count);

  // Basic validation
  $clean = [];
  foreach ($questions as $idx => $q) {
    if (!is_array($q)) continue;
    $qp = trim((string)($q['prompt'] ?? ''));
    $opts = $q['options'] ?? [];
    $ai = (int)($q['answerIndex'] ?? -1);
    $ex = trim((string)($q['explanation'] ?? ''));
    if ($qp === '' || !is_array($opts) || count($opts) !== 4) continue;
    $opts = array_map(fn($v) => trim((string)$v), $opts);
    if ($ai < 0 || $ai > 3) continue;
    $clean[] = [
      'prompt' => $qp,
      'options' => $opts,
      'answerIndex' => $ai,
      'explanation' => $ex === '' ? null : $ex,
    ];
  }

  if (count($clean) < 3) {
    json_response(['success' => false, 'error' => 'AI did not generate enough valid questions'], 502);
  }

  // Save sanitized questions to cache (2 weeks)
  ai_cache_set(
    $pdo,
    $cacheKey,
    'generate_bank',
    gemini_model(),
    json_encode(['questions' => $clean], JSON_UNESCAPED_SLASHES),
    60 * 60 * 24 * 14
  );
}

try {
  $pdo->beginTransaction();

  $b = $pdo->prepare('INSERT INTO banks (id, course_id, title, mode, created_by, created_at)
                      VALUES (:id, :cid, :title, :mode, :by, UTC_TIMESTAMP())');
  $b->execute([
    ':id' => $bankId,
    ':cid' => $courseId,
    ':title' => $bankTitle,
    ':mode' => 'mcq',
    ':by' => $me['id'],
  ]);

  $insQ = $pdo->prepare('INSERT INTO questions (bank_id, id, prompt, `options`, answer_index, explanation, sort_order)
                         VALUES (:bid, :id, :p, :o, :ai, :e, :s)');
  $order = 1;
  foreach ($clean as $i => $q) {
    $qid = $bankId . '_q' . str_pad((string)($i + 1), 3, '0', STR_PAD_LEFT);
    $insQ->execute([
      ':bid' => $bankId,
      ':id' => $qid,
      ':p' => $q['prompt'],
      ':o' => json_encode($q['options'], JSON_UNESCAPED_SLASHES),
      ':ai' => (int)$q['answerIndex'],
      ':e' => $q['explanation'],
      ':s' => $order++,
    ]);
  }

  // Optional usage log
  try {
    $h = hash('sha256', json_encode(['courseId' => $courseId, 'topic' => $topic, 'difficulty' => $difficulty, 'count' => $count]));
    $log = $pdo->prepare('INSERT INTO ai_request_log (user_id, kind, prompt_hash, created_at) VALUES (:uid, :k, :h, UTC_TIMESTAMP())');
    $log->execute([':uid' => $me['id'], ':k' => 'generate_bank', ':h' => $h]);
  } catch (Throwable $e) {
    // ignore
  }

  $pdo->commit();
} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  json_response(['success' => false, 'error' => 'Failed to save generated bank', 'detail' => $e->getMessage()], 500);
}

json_response([
  'success' => true,
  'data' => [
    'bankId' => $bankId,
    'title' => $bankTitle,
    'questionCount' => count($clean),
    'cached' => $cachedHit,
  ],
]);
