<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

require_once __DIR__ . '/_gemini.php';
require_once __DIR__ . '/_limits.php';
require_once __DIR__ . '/_cache.php';

/**
 * THEORY AI COACH
 *
 * Student-facing tool for:
 * - hint mode: outline + hint before writing
 * - feedback mode: feedback after writing (optional score)
 *
 * Uses:
 * - ai_request_log (rate limits)
 * - ai_cache (cache) when available
 */

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$body = read_json();
require_fields($body, ['bankId', 'questionId', 'mode']);

$bankId = trim((string)($body['bankId'] ?? ''));
$questionId = trim((string)($body['questionId'] ?? ''));
$mode = strtolower(trim((string)($body['mode'] ?? '')));
$answerText = trim((string)($body['answerText'] ?? ''));

// Support both param names (frontend might send wantScore)
$includeScore = isset($body['includeScore'])
  ? (bool)$body['includeScore']
  : (isset($body['wantScore']) ? (bool)$body['wantScore'] : true);

$attemptId = trim((string)($body['attemptId'] ?? ''));

if ($bankId === '' || $questionId === '') {
  json_response(['success' => false, 'error' => 'Missing bankId/questionId'], 422);
}

if (!in_array($mode, ['hint', 'feedback'], true)) {
  json_response(['success' => false, 'error' => 'Invalid mode. Use hint|feedback'], 422);
}

if ($mode === 'feedback' && $answerText === '') {
  json_response(['success' => false, 'error' => 'Answer text is required for feedback mode'], 422);
}

// Bank + course context
$b = $pdo->prepare('SELECT b.id, b.course_id AS courseId, b.title AS bankTitle, b.mode,
                           c.code AS courseCode, c.title AS courseTitle
                    FROM banks b
                    LEFT JOIN courses c ON c.id = b.course_id
                    WHERE b.id = :id
                    LIMIT 1');
$b->execute([':id' => $bankId]);
$bank = $b->fetch();

if (!$bank) {
  json_response(['success' => false, 'error' => 'Bank not found'], 404);
}

if (strtolower((string)($bank['mode'] ?? '')) !== 'theory') {
  json_response(['success' => false, 'error' => 'This bank is not a theory bank'], 422);
}

// Course access check (admins bypass)
$courseId = (string)($bank['courseId'] ?? '');
if ($courseId !== '' && (string)($me['role'] ?? '') !== 'admin') {
  try {
    $chk = $pdo->prepare('SELECT 1 FROM user_courses WHERE user_id = :uid AND course_id = :cid LIMIT 1');
    $chk->execute([':uid' => $me['id'], ':cid' => $courseId]);
    if (!(bool)$chk->fetchColumn()) {
      json_response(['success' => false, 'error' => 'You do not have access to this course'], 403);
    }
  } catch (Throwable $e) {
    // If user_courses table isn't present, don't hard-fail.
  }
}

// Load question
$q = $pdo->prepare('SELECT id, prompt, guide, points_json AS pointsJson
                    FROM theory_questions
                    WHERE id = :qid AND bank_id = :bid
                    LIMIT 1');
$q->execute([':qid' => $questionId, ':bid' => $bankId]);
$qr = $q->fetch();
if (!$qr) {
  json_response(['success' => false, 'error' => 'Question not found in this bank'], 404);
}

$promptText = trim((string)($qr['prompt'] ?? ''));
$guide = $qr['guide'] === null ? '' : trim((string)$qr['guide']);
$points = [];
if (!empty($qr['pointsJson'])) {
  try {
    $tmp = json_decode((string)$qr['pointsJson'], true);
    if (is_array($tmp)) $points = $tmp;
  } catch (Throwable $e) {
    $points = [];
  }
}

// Clean up points
if (count($points) > 12) $points = array_slice($points, 0, 12);
$points = array_values(array_filter(array_map(static function ($v) {
  $s = trim((string)$v);
  return $s === '' ? null : $s;
}, $points)));

if ($promptText === '') {
  json_response(['success' => false, 'error' => 'This theory question has no prompt'], 409);
}

$courseCode = (string)($bank['courseCode'] ?? $bank['courseId'] ?? '');
$courseTitle = (string)($bank['courseTitle'] ?? '');
$bankTitle = (string)($bank['bankTitle'] ?? '');

// Hashes (so edits invalidate cache)
$qHash = hash('sha256', json_encode([
  'prompt' => $promptText,
  'guide' => $guide,
  'points' => $points,
], JSON_UNESCAPED_SLASHES));
$aHash = $mode === 'feedback' ? hash('sha256', $answerText) : '';

$cacheKey = ai_cache_key('theory_coach', [
  'bankId' => $bankId,
  'questionId' => $questionId,
  'mode' => $mode,
  // cache feedback per-user to avoid cross-user leakage
  'uid' => $mode === 'feedback' ? (string)$me['id'] : '',
  'includeScore' => $includeScore ? 1 : 0,
  'qHash' => $qHash,
  'aHash' => $aHash,
  'model' => gemini_model(),
]);

$cachedJson = ai_cache_get($pdo, $cacheKey);
if (is_string($cachedJson)) {
  $tmp = json_decode($cachedJson, true);
  if (is_array($tmp) && isset($tmp['mode']) && isset($tmp['result'])) {
    // best-effort store to attempt if provided
    if ($mode === 'feedback' && $attemptId !== '') {
      try {
        $st = $pdo->prepare('UPDATE user_theory_attempts
                             SET ai_score = :s,
                                 ai_feedback_json = :j,
                                 ai_model = :m,
                                 ai_updated_at = UTC_TIMESTAMP()
                             WHERE id = :id AND user_id = :uid AND bank_id = :bid AND question_id = :qid
                             LIMIT 1');
        $scoreOrNull = $tmp['result']['aiScore'] ?? null;
        if ($scoreOrNull === '') $scoreOrNull = null;
        $st->execute([
          ':s' => $scoreOrNull,
          ':j' => json_encode($tmp['result'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
          ':m' => (string)($tmp['model'] ?? gemini_model()),
          ':id' => $attemptId,
          ':uid' => (string)$me['id'],
          ':bid' => $bankId,
          ':qid' => $questionId,
        ]);
      } catch (Throwable $e) {
        // ignore if columns/table aren't migrated yet
      }
    }

    json_response([
      'success' => true,
      'data' => array_merge($tmp, [
        'model' => (string)($tmp['model'] ?? gemini_model()),
        'cached' => true,
      ]),
    ]);
  }
}

// Rate limits only for real model calls
ai_enforce_limits($pdo, (string)$me['id'], 'theory_coach');
$callHash = hash('sha256', json_encode([
  'bankId' => $bankId,
  'questionId' => $questionId,
  'mode' => $mode,
  'includeScore' => $includeScore,
  'qHash' => $qHash,
  'aHash' => $aHash,
], JSON_UNESCAPED_SLASHES));
ai_log_call($pdo, (string)$me['id'], 'theory_coach', $callHash);

// Prompt building helpers
$clamp = static function (string $s, int $maxChars): string {
  $s = trim($s);
  if ($s === '') return '';
  if ($maxChars < 200) $maxChars = 200;

  $len = function_exists('mb_strlen') ? (int)mb_strlen($s, 'UTF-8') : strlen($s);
  if ($len <= $maxChars) return $s;

  $cut = function_exists('mb_substr') ? (string)mb_substr($s, 0, $maxChars, 'UTF-8') : substr($s, 0, $maxChars);
  return rtrim($cut) . '…';
};

$header = "You are a calm, accurate tutor for Nigerian university students.\n"
  . "Safety rules: do not include hate/harassment/sexual content; do not provide illegal instructions; do not request personal data.\n"
  . "Course: {$courseCode}" . ($courseTitle !== '' ? " ({$courseTitle})" : '') . "\n"
  . "Bank: {$bankTitle}\n\n"
  . "THEORY QUESTION:\n" . $promptText . "\n\n";

$marking = '';
if ($guide !== '' || count($points) > 0) {
  $marking .= "MARKING GUIDE (may be incomplete):\n";
  if ($guide !== '') {
    $marking .= "Model guide:\n" . $clamp($guide, 2500) . "\n\n";
  }
  if (count($points) > 0) {
    $marking .= "Key points (marking points):\n- " . implode("\n- ", array_map(static fn($p) => $clamp((string)$p, 220), $points)) . "\n\n";
  }
}

if ($mode === 'hint') {
  $prompt = $header
    . $marking
    . "Task: Give a hint + outline that helps the student write a strong answer, WITHOUT writing the full answer.\n"
    . "Return ONLY valid JSON (no markdown, no extra text) in this exact shape:\n"
    . '{"mode":"hint","result":{"hint":"...","outline":["..."],"commonMistakes":["..."],"keywords":["..."]}}' . "\n\n"
    . "Rules:\n"
    . "- Do NOT write a full model answer. Provide a plan/outline only.\n"
    . "- outline: 4-7 short bullets.\n"
    . "- hint: 2-5 lines of strategy (how to approach the answer).\n"
    . "- commonMistakes: 2-6 bullets.\n"
    . "- keywords: 4-10 key terms the student should mention (prefer using the provided points if available).\n";

  $resp = gemini_generate_text($prompt, [
    'temperature' => 0.35,
    'maxOutputTokens' => 900,
  ]);

  if (!$resp['ok']) {
    json_response(['success' => false, 'error' => $resp['error'] ?? 'AI failed'], (int)($resp['status'] ?? 502));
  }

  $text = trim((string)$resp['text']);
  $jsonText = ai_extract_json($text);
  $data = $jsonText !== null ? json_decode($jsonText, true) : null;

  if (!is_array($data) || !isset($data['result']) || !is_array($data['result'])) {
    json_response(['success' => false, 'error' => 'AI returned invalid format'], 502);
  }

  $r = $data['result'];

  $cleanList = static function ($arr, int $maxItems) {
    $out = [];
    if (!is_array($arr)) return $out;
    foreach ($arr as $v) {
      $s = trim((string)$v);
      if ($s === '') continue;
      $out[] = $s;
      if (count($out) >= $maxItems) break;
    }
    return $out;
  };

  $out = [
    'mode' => 'hint',
    'result' => [
      'hint' => trim((string)($r['hint'] ?? '')),
      'outline' => $cleanList($r['outline'] ?? [], 8),
      'commonMistakes' => $cleanList($r['commonMistakes'] ?? [], 10),
      'keywords' => $cleanList($r['keywords'] ?? [], 12),
    ],
    'model' => gemini_model(),
    'cached' => false,
  ];

  ai_cache_set($pdo, $cacheKey, 'theory_coach', gemini_model(), json_encode($out, JSON_UNESCAPED_SLASHES), 60 * 60 * 24 * 7);
  json_response(['success' => true, 'data' => $out]);
}

// feedback mode
$prompt = $header
  . $marking
  . "STUDENT ANSWER:\n" . $clamp($answerText, 4200) . "\n\n"
  . "Task: Give constructive feedback on the student's answer.\n"
  . "Return ONLY valid JSON (no markdown, no extra text) in this exact shape:\n"
  . '{"mode":"feedback","result":{"aiScore":0,"summary":"...","strengths":["..."],"missingPoints":["..."],"improvements":["..."],"rewriteSuggestion":"..."}}' . "\n\n"
  . "Rules:\n"
  . "- Be specific and helpful.\n"
  . "- summary: 2-5 lines summarizing the main feedback.\n"
  . "- strengths: 2-5 bullets.\n"
  . "- missingPoints: 2-8 bullets (compare to marking points if provided).\n"
  . "- improvements: 2-6 bullets.\n"
  . "- rewriteSuggestion: 4-8 lines showing a better structure (not a perfect long essay).\n"
  . ($includeScore
      ? "- aiScore is 0..5 based on coverage + clarity + correctness.\n"
      : "- Set aiScore to null.\n");

$resp = gemini_generate_text($prompt, [
  'temperature' => 0.3,
  'maxOutputTokens' => 1200,
]);

if (!$resp['ok']) {
  json_response(['success' => false, 'error' => $resp['error'] ?? 'AI failed'], (int)($resp['status'] ?? 502));
}

$text = trim((string)$resp['text']);
$jsonText = ai_extract_json($text);
$data = $jsonText !== null ? json_decode($jsonText, true) : null;

if (!is_array($data) || !isset($data['result']) || !is_array($data['result'])) {
  json_response(['success' => false, 'error' => 'AI returned invalid format'], 502);
}

$r = $data['result'];

$cleanList = static function ($arr, int $maxItems) {
  $out = [];
  if (!is_array($arr)) return $out;
  foreach ($arr as $v) {
    $s = trim((string)$v);
    if ($s === '') continue;
    $out[] = $s;
    if (count($out) >= $maxItems) break;
  }
  return $out;
};

$aiScore = null;
if ($includeScore) {
  $s = $r['aiScore'] ?? null;
  if ($s !== null && $s !== '') {
    $aiScore = (int)$s;
    if ($aiScore < 0) $aiScore = 0;
    if ($aiScore > 5) $aiScore = 5;
  }
}

$out = [
  'mode' => 'feedback',
  'result' => [
    'aiScore' => $aiScore,
    'summary' => trim((string)($r['summary'] ?? '')),
    'strengths' => $cleanList($r['strengths'] ?? [], 8),
    'missingPoints' => $cleanList($r['missingPoints'] ?? [], 12),
    'improvements' => $cleanList($r['improvements'] ?? [], 10),
    'rewriteSuggestion' => trim((string)($r['rewriteSuggestion'] ?? '')),
  ],
  'model' => gemini_model(),
  'cached' => false,
];

// Best-effort store to attempt if provided
if ($attemptId !== '') {
  try {
    $st = $pdo->prepare('UPDATE user_theory_attempts
                         SET ai_score = :s,
                             ai_feedback_json = :j,
                             ai_model = :m,
                             ai_updated_at = UTC_TIMESTAMP()
                         WHERE id = :id AND user_id = :uid AND bank_id = :bid AND question_id = :qid
                         LIMIT 1');
    $st->execute([
      ':s' => $aiScore,
      ':j' => json_encode($out['result'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
      ':m' => gemini_model(),
      ':id' => $attemptId,
      ':uid' => (string)$me['id'],
      ':bid' => $bankId,
      ':qid' => $questionId,
    ]);
  } catch (Throwable $e) {
    // ignore if columns/table aren't migrated yet
  }
}

ai_cache_set($pdo, $cacheKey, 'theory_coach', gemini_model(), json_encode($out, JSON_UNESCAPED_SLASHES), 60 * 60 * 24 * 7);

json_response(['success' => true, 'data' => $out]);
