<?php
// public_html/jabuspark_api/auth.php
declare(strict_types=1);

require_once __DIR__ . '/response.php';
require_once __DIR__ . '/helpers.php';

/**
 * Sessions are opaque tokens stored hashed in DB.
 * Client sends: Authorization: Bearer <token>
 */

function token_hash(string $token): string {
  // constant-time hash; store hex
  return hash('sha256', $token, false);
}

function create_session(PDO $pdo, array $user): array {
  $token = bin2hex(random_bytes(32));
  $hash = token_hash($token);
  $sid = uuidv4();
  $now = new DateTimeImmutable('now', new DateTimeZone('UTC'));
  $expires = $now->modify('+' . SESSION_TTL_DAYS . ' days');

  $stmt = $pdo->prepare('INSERT INTO sessions (id, user_id, token_hash, created_at, expires_at, ip, user_agent)
                         VALUES (:id, :uid, :th, :ca, :ea, :ip, :ua)');
  $stmt->execute([
    ':id' => $sid,
    ':uid' => $user['id'],
    ':th' => $hash,
    ':ca' => $now->format('Y-m-d H:i:s'),
    ':ea' => $expires->format('Y-m-d H:i:s'),
    ':ip' => substr((string)($_SERVER['REMOTE_ADDR'] ?? ''), 0, 64),
    ':ua' => substr((string)($_SERVER['HTTP_USER_AGENT'] ?? ''), 0, 255),
  ]);

  return ['token' => $token, 'expiresAt' => $expires->format('c')];
}

function find_user_by_session(PDO $pdo, string $token): ?array {
  $hash = token_hash($token);
  $stmt = $pdo->prepare('SELECT u.* FROM sessions s
                         JOIN users u ON u.id = s.user_id
                         WHERE s.token_hash = :th AND s.expires_at > UTC_TIMESTAMP()
                         LIMIT 1');
  $stmt->execute([':th' => $hash]);
  $u = $stmt->fetch();
  return $u ?: null;
}

function require_auth(PDO $pdo): array {
  $token = bearer_token();
  if (!$token) json_response(['success' => false, 'error' => 'Unauthorized'], 401);

  $u = find_user_by_session($pdo, $token);
  if (!$u) json_response(['success' => false, 'error' => 'Invalid or expired session'], 401);

  return $u;
}

function require_admin(array $user): void {
  if (($user['role'] ?? '') !== 'admin') {
    json_response(['success' => false, 'error' => 'Admin only'], 403);
  }
}

/**
 * Uploader permission:
 * - admin can upload anything
 * - course_rep can upload only for assigned courses
 */
function require_uploader(PDO $pdo, array $user, string $courseId): void {
  $role = (string)($user['role'] ?? '');
  if ($role === 'admin') return;

  if ($role !== 'course_rep') {
    json_response(['success' => false, 'error' => 'Uploader access required'], 403);
  }

  // soft flag: allow admins to temporarily disable a rep's upload access
  try {
    if (rep_uploads_disabled($pdo, (string)$user['id'])) {
      json_response(['success' => false, 'error' => 'Uploads are disabled for your account. Contact an admin.'], 403);
    }
  } catch (Throwable $e) {
    // ignore if flag table doesn't exist yet
  }

  $cid = trim((string)$courseId);
  if ($cid === '') {
    json_response(['success' => false, 'error' => 'Missing courseId'], 422);
  }

  $stmt = $pdo->prepare('SELECT 1 FROM course_rep_courses WHERE user_id = :uid AND course_id = :cid LIMIT 1');
  $stmt->execute([':uid' => $user['id'], ':cid' => $cid]);
  $ok = (bool)$stmt->fetchColumn();
  if (!$ok) {
    json_response(['success' => false, 'error' => 'Not allowed to upload for this course'], 403);
  }
}

/**
 * Optional flag table (created automatically if missing).
 * This keeps the workflow stable even if you haven't run a migration.
 */
function ensure_course_rep_flags_table(PDO $pdo): void {
  $pdo->exec('CREATE TABLE IF NOT EXISTS course_rep_flags (
    user_id CHAR(36) NOT NULL,
    uploads_disabled TINYINT(1) NOT NULL DEFAULT 0,
    updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (user_id)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci');
}

function rep_uploads_disabled(PDO $pdo, string $userId): bool {
  $uid = trim($userId);
  if ($uid === '') return false;
  ensure_course_rep_flags_table($pdo);
  $stmt = $pdo->prepare('SELECT uploads_disabled FROM course_rep_flags WHERE user_id = :uid LIMIT 1');
  $stmt->execute([':uid' => $uid]);
  $v = $stmt->fetchColumn();
  return (int)($v ?? 0) === 1;
}

function safe_user(PDO $pdo, array $user): array {
  // Attach profile + courses
  $profile = [
    'facultyId' => null,
    'departmentId' => null,
    'level' => null,
    'courseIds' => [],
    'repCourseIds' => [],
  ];

  $stmt = $pdo->prepare('SELECT faculty_id, department_id, level FROM profiles WHERE user_id = :uid LIMIT 1');
  $stmt->execute([':uid' => $user['id']]);
  $p = $stmt->fetch();
  if ($p) {
    $profile['facultyId'] = $p['faculty_id'];
    $profile['departmentId'] = $p['department_id'];
    $profile['level'] = $p['level'] !== null ? (int)$p['level'] : null;
  }

  $stmt = $pdo->prepare('SELECT course_id FROM user_courses WHERE user_id = :uid');
  $stmt->execute([':uid' => $user['id']]);
  $profile['courseIds'] = array_values(array_map(fn($r) => $r['course_id'], $stmt->fetchAll() ?: []));

  // course rep upload permissions
  try {
    $stmt = $pdo->prepare('SELECT course_id FROM course_rep_courses WHERE user_id = :uid');
    $stmt->execute([':uid' => $user['id']]);
    $profile['repCourseIds'] = array_values(array_map(fn($r) => $r['course_id'], $stmt->fetchAll() ?: []));
  } catch (Throwable $e) {
    // table may not exist yet during migration; keep empty
    $profile['repCourseIds'] = [];
  }

  // course rep flags (optional)
  $uploadsDisabled = false;
  try {
    $uploadsDisabled = rep_uploads_disabled($pdo, (string)$user['id']);
  } catch (Throwable $e) {
    $uploadsDisabled = false;
  }

  // latest course rep request status (optional, for Profile workflow)
  $repRequest = null;
  try {
    $stmt = $pdo->prepare('SELECT id, status, requested_at, reviewed_at, admin_note
                           FROM course_rep_requests
                           WHERE user_id = :uid
                           ORDER BY requested_at DESC
                           LIMIT 1');
    $stmt->execute([':uid' => $user['id']]);
    $r = $stmt->fetch();
    if ($r) {
      $repRequest = [
        'id' => $r['id'],
        'status' => $r['status'],
        'requestedAt' => $r['requested_at'],
        'reviewedAt' => $r['reviewed_at'],
        'adminNote' => $r['admin_note'],
      ];
    }
  } catch (Throwable $e) {
    $repRequest = null;
  }

  return [
    'id' => $user['id'],
    'email' => $user['email'],
    'fullName' => $user['full_name'],
    'role' => $user['role'],
    'createdAt' => $user['created_at'],
    'uploadsDisabled' => $uploadsDisabled,
    'repRequest' => $repRequest,
    'profile' => $profile
  ];
}
