<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

$notify_helpers = __DIR__ . '/../notify/_helpers.php';
if (is_file($notify_helpers)) { require_once $notify_helpers; }

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);

$body = read_json();
require_fields($body, ['courseId', 'title', 'mode', 'questions']);

$courseId = trim((string)$body['courseId']);
$title = trim((string)$body['title']);
$mode = strtolower(trim((string)$body['mode']));
$questions = $body['questions'];

if ($courseId === '' || $title === '' || $mode === '') {
  json_response(['success' => false, 'error' => 'Invalid payload'], 422);
}

if (!in_array($mode, ['mcq', 'theory'], true)) {
  json_response(['success' => false, 'error' => 'mode must be "mcq" or "theory"'], 422);
}

require_uploader($pdo, $me, $courseId);
if (!is_array($questions) || count($questions) < 1) {
  json_response(['success' => false, 'error' => 'questions must be a non-empty array'], 422);
}

$bid = uuidv4();

try {
  $pdo->beginTransaction();

  $pdo->prepare('INSERT INTO banks (id, course_id, title, mode, created_by, created_at)
                 VALUES (:id,:c,:t,:m,:cb,UTC_TIMESTAMP())')
      ->execute([':id' => $bid, ':c' => $courseId, ':t' => $title, ':m' => $mode, ':cb' => $me['id']]);

  if ($mode === 'theory') {
    // -----------------------------
    // Theory questions
    // -----------------------------
    $ins = $pdo->prepare('INSERT INTO theory_questions (id, bank_id, prompt, guide, points_json, sort_order)
                          VALUES (:id,:bid,:p,:g,:pts,:so)');

    $i = 0;
    foreach ($questions as $q) {
      $i++;
      $qid = isset($q['id']) && (string)$q['id'] !== '' ? (string)$q['id'] : uuidv4();
      $prompt = trim((string)($q['prompt'] ?? $q['question'] ?? ''));
      $guide = isset($q['guide']) ? trim((string)$q['guide']) : (isset($q['modelAnswer']) ? trim((string)$q['modelAnswer']) : (isset($q['answer']) ? trim((string)$q['answer']) : ''));

      // points can be: points | markingPoints | keyPoints
      $points = $q['points'] ?? $q['markingPoints'] ?? $q['keyPoints'] ?? null;
      $ptsArr = null;
      if (is_string($points)) {
        $lines = array_values(array_filter(array_map('trim', preg_split('/\r?\n/', $points))));
        $ptsArr = $lines;
      } elseif (is_array($points)) {
        $ptsArr = array_values(array_filter(array_map('strval', $points), static fn($v) => trim($v) !== ''));
      }

      if ($prompt === '') {
        json_response(['success' => false, 'error' => 'Invalid theory question at index ' . ($i - 1)], 422);
      }

      $ins->execute([
        ':id' => $qid,
        ':bid' => $bid,
        ':p' => $prompt,
        ':g' => $guide !== '' ? $guide : null,
        ':pts' => $ptsArr !== null ? json_encode($ptsArr, JSON_UNESCAPED_UNICODE) : null,
        ':so' => $i,
      ]);
    }
  } else {
    // -----------------------------
    // MCQ questions
    // -----------------------------
    $ins = $pdo->prepare('INSERT INTO questions (id, bank_id, prompt, options, answer_index, explanation, sort_order)
                          VALUES (:id,:bid,:p,:o,:ai,:ex,:so)');

    $i = 0;
    foreach ($questions as $q) {
      $i++;
      $qid = isset($q['id']) && (string)$q['id'] !== '' ? (string)$q['id'] : uuidv4();
      $prompt = trim((string)($q['prompt'] ?? ''));
      $options = $q['options'] ?? [];
      $answerIndex = (int)($q['answerIndex'] ?? 0);
      $explain = (string)($q['explain'] ?? '');

      if ($prompt === '' || !is_array($options) || count($options) < 2) {
        json_response(['success' => false, 'error' => 'Invalid question at index ' . ($i - 1)], 422);
      }
      if ($answerIndex < 0 || $answerIndex >= count($options)) {
        json_response(['success' => false, 'error' => 'answerIndex out of range at index ' . ($i - 1)], 422);
      }

      $ins->execute([
        ':id' => $qid,
        ':bid' => $bid,
        ':p' => $prompt,
        ':o' => json_encode(array_values($options), JSON_UNESCAPED_UNICODE),
        ':ai' => $answerIndex,
        ':ex' => $explain,
        ':so' => $i,
      ]);
    }
  }

  $pdo->commit();

  // Best-effort: create an announcement post for this bank
  try {
    if (function_exists('notify_ensure_course_channel')) {
      $ch = notify_ensure_course_channel($pdo, $courseId, $me['id']);
      if ($ch && isset($ch['id'])) {
        $pid = uuidv4();
        $ptitle = ($mode === 'theory') ? ('New theory bank: ' . $title) : ('New practice bank: ' . $title);
        $pbody = ($mode === 'theory')
          ? ('A new theory bank was created for ' . strtoupper($courseId) . '. Practice writing + compare with the marking guide.')
          : ('A new practice bank was created for ' . strtoupper($courseId) . '. Jump in and start practising.');

        $insP = $pdo->prepare('INSERT INTO notify_posts (id, channel_id, title, body, link_url, attachment_url, attachment_type, is_pinned, created_by)
                               VALUES (:id,:cid,:t,:b,NULL,NULL,NULL,0,:uid)');
        $insP->execute([
          ':id' => $pid,
          ':cid' => (string)$ch['id'],
          ':t' => $ptitle,
          ':b' => $pbody,
          ':uid' => $me['id'],
        ]);
      }
    }
  } catch (Throwable $e) {
    // ignore
  }

  json_response(['success' => true, 'data' => ['bankId' => $bid]], 201);
} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  json_response(['success' => false, 'error' => 'Failed to create bank'], 500);
}
