<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'GET') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$id = (string)($_GET['id'] ?? '');
if ($id === '') {
  json_response(['success' => false, 'error' => 'Missing id'], 422);
}

try {
  // Bank
  $stmt = $pdo->prepare(
    'SELECT id, course_id AS courseId, title, mode
     FROM banks
     WHERE id = :id
     LIMIT 1'
  );
  $stmt->execute([':id' => $id]);
  $bank = $stmt->fetch();

  if (!$bank) {
    json_response(['success' => false, 'error' => 'Not found'], 404);
  }

  $mode = strtolower((string)($bank['mode'] ?? 'mcq'));

  // -----------------------------
  // Theory banks
  // -----------------------------
  if ($mode === 'theory') {
    $questions = [];
    try {
      $q = $pdo->prepare(
        'SELECT id,
                prompt,
                guide,
                points_json AS pointsJson,
                sort_order AS sortOrder
         FROM theory_questions
         WHERE bank_id = :id
         ORDER BY sort_order ASC, id ASC'
      );
      $q->execute([':id' => $id]);
      $rows = $q->fetchAll() ?: [];

      $questions = array_map(static function ($r) {
        $pts = [];
        $raw = $r['pointsJson'] ?? null;
        if ($raw !== null && $raw !== '') {
          $d = json_decode((string)$raw, true);
          if (is_array($d)) $pts = $d;
        }

        return [
          'id' => (string)($r['id'] ?? ''),
          'prompt' => (string)($r['prompt'] ?? ''),
          'guide' => $r['guide'] === null ? null : (string)$r['guide'],
          'points' => $pts,
          'sortOrder' => (int)($r['sortOrder'] ?? 1),
        ];
      }, $rows);
    } catch (Throwable $e) {
      // If theory tables aren't migrated yet, return bank with zero questions.
      $questions = [];
    }

    $bank['questions'] = $questions;

    json_response([
      'success' => true,
      'data' => [
        'bank' => $bank,
      ],
    ]);
  }

  // -----------------------------
  // MCQ banks
  // -----------------------------

  // Questions (use backticks to avoid any reserved-word issues)
  $q = $pdo->prepare(
    'SELECT id,
            prompt,
            `options` AS options,
            answer_index AS answerIndex,
            explanation AS explanation,
            sort_order AS sortOrder
     FROM questions
     WHERE bank_id = :id
     ORDER BY sort_order ASC, id ASC'
  );
  $q->execute([':id' => $id]);

  $rows = $q->fetchAll() ?: [];

  $questions = array_map(static function ($r) {
    $opts = json_decode((string)($r['options'] ?? '[]'), true);
    if (!is_array($opts)) $opts = [];

    return [
      'id' => (string)$r['id'],
      'prompt' => (string)$r['prompt'],
      'options' => $opts,
      'answerIndex' => (int)($r['answerIndex'] ?? 0),
      'explanation' => $r['explanation'] === null ? null : (string)$r['explanation'],
      'sortOrder' => (int)($r['sortOrder'] ?? 1),
    ];
  }, $rows);

  $bank['questions'] = $questions;

  json_response([
    'success' => true,
    'data' => [
      'bank' => $bank,
    ],
  ]);

} catch (Throwable $e) {
  // For debugging on shared hosting, return the real SQL/PHP error.
  // Once fixed, you can change this back to a generic "Server error".
  json_response([
    'success' => false,
    'error' => 'Server error',
    'detail' => $e->getMessage(),
  ], 500);
}
