<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'GET') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$courseId = isset($_GET['courseId']) ? (string)$_GET['courseId'] : '';

/**
 * questionCount logic:
 * - MCQ banks use the `questions` table.
 * - Theory banks use the `theory_questions` table.
 *
 * If the theory migration hasn't been applied yet, the query may fail.
 * We fall back gracefully so existing MCQ functionality never breaks.
 */
$sqlMain =
  'SELECT b.id,
          b.course_id AS courseId,
          b.title,
          b.mode,
          CASE
            WHEN b.mode = "theory" THEN (SELECT COUNT(*) FROM theory_questions tq WHERE tq.bank_id = b.id)
            ELSE (SELECT COUNT(*) FROM questions q WHERE q.bank_id = b.id)
          END AS questionCount
   FROM banks b';

$sqlFallback =
  'SELECT b.id,
          b.course_id AS courseId,
          b.title,
          b.mode,
          (SELECT COUNT(*) FROM questions q WHERE q.bank_id = b.id) AS questionCount
   FROM banks b';

try {
  $sql = $sqlMain;

  if ($courseId !== '') {
    $sql .= ' WHERE b.course_id = :c';
  }

  // Prefer newest first. (If you have a created_at column, feel free to order by it.)
  $sql .= ' ORDER BY b.id DESC';

  if ($courseId !== '') {
    $stmt = $pdo->prepare($sql);
    $stmt->execute([':c' => $courseId]);
    $rows = $stmt->fetchAll() ?: [];
  } else {
    $rows = $pdo->query($sql)->fetchAll() ?: [];
  }
} catch (Throwable $e) {
  // Theory tables may not exist yet; keep MCQ list working.
  $sql = $sqlFallback;

  if ($courseId !== '') {
    $sql .= ' WHERE b.course_id = :c';
  }

  $sql .= ' ORDER BY b.id DESC';

  if ($courseId !== '') {
    $stmt = $pdo->prepare($sql);
    $stmt->execute([':c' => $courseId]);
    $rows = $stmt->fetchAll() ?: [];
  } else {
    $rows = $pdo->query($sql)->fetchAll() ?: [];
  }
}

$banks = array_map(static function ($r) {
  $r['questionCount'] = (int)($r['questionCount'] ?? 0);
  return $r;
}, $rows);

json_response([
  'success' => true,
  'data' => [
    'banks' => $banks,
  ],
]);
