<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../groups/_helpers.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$body = read_json();
require_fields($body, ['challengeId', 'answers']);

$challengeId = (string)$body['challengeId'];
$answers = $body['answers']; // { [questionId]: selectedIndex }
$secondsTotal = isset($body['secondsTotal']) ? (int)$body['secondsTotal'] : 0;

if (!is_array($answers)) json_response(['success' => false, 'error' => 'answers must be an object/map'], 422);

$ch = challenge_require_access($pdo, $challengeId, (string)$me['id']);
$groupId = (string)$ch['group_id'];
$courseId = (string)$ch['course_id'];

$qq = $pdo->prepare('SELECT question_id AS questionId, bank_id AS bankId, position FROM study_group_challenge_questions WHERE challenge_id = :id ORDER BY position ASC');
$qq->execute(['id' => $challengeId]);
$challengeQuestions = $qq->fetchAll() ?: [];
if (count($challengeQuestions) === 0) json_response(['success' => false, 'error' => 'Challenge has no questions'], 500);

$qids = array_map(fn($r) => (string)$r['questionId'], $challengeQuestions);
$in = implode(',', array_fill(0, count($qids), '?'));
$stmt = $pdo->prepare('
  SELECT q.id, q.bank_id AS bankId, q.prompt, q.options, q.answer_index AS answerIndex, q.explanation AS explanationText
  FROM questions q
  WHERE q.id IN (' . $in . ')
');
$stmt->execute($qids);
$rows = $stmt->fetchAll() ?: [];
$byId = [];
foreach ($rows as $r) $byId[(string)$r['id']] = $r;

$total = count($challengeQuestions);
$correct = 0;
$reviewWrong = [];

$insAttempt = $pdo->prepare('
  INSERT INTO user_question_attempts (user_id, bank_id, question_id, selected_index, is_correct, seconds_spent, created_at)
  VALUES (:uid, :bid, :qid, :sel, :isc, :sec, UTC_TIMESTAMP())
');

$uwqSel = $pdo->prepare('SELECT status, repetitions, interval_days, ease, wrong_count FROM user_wrong_questions WHERE user_id = :uid AND bank_id = :bid AND question_id = :qid LIMIT 1');
$uwqIns = $pdo->prepare('
  INSERT INTO user_wrong_questions
    (user_id, course_id, bank_id, question_id, status, repetitions, interval_days, ease, wrong_count, last_review_at, next_review_at, created_at, updated_at)
  VALUES
    (:uid, :cid, :bid, :qid, "active", 0, 1, 2.30, 1, UTC_TIMESTAMP(), :n, UTC_TIMESTAMP(), UTC_TIMESTAMP())
');
$uwqUpd = $pdo->prepare('
  UPDATE user_wrong_questions
  SET course_id = :cid,
      status = :st,
      repetitions = :r,
      interval_days = :i,
      ease = :e,
      wrong_count = :w,
      last_review_at = UTC_TIMESTAMP(),
      next_review_at = :n,
      updated_at = UTC_TIMESTAMP()
  WHERE user_id = :uid AND bank_id = :bid AND question_id = :qid
');
$now = new DateTimeImmutable('now', new DateTimeZone('UTC'));

try {
  $pdo->beginTransaction();

  // block double submit
  $dupe = $pdo->prepare('SELECT id FROM study_group_challenge_attempts WHERE challenge_id = :cid AND user_id = :uid LIMIT 1');
  $dupe->execute([':cid' => $challengeId, ':uid' => $me['id']]);
  if ($dupe->fetch()) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    json_response(['success' => false, 'error' => 'You already submitted this challenge'], 409);
  }

  $per = $secondsTotal > 0 ? (int)floor($secondsTotal / max(1, $total)) : 0;

  foreach ($challengeQuestions as $posRow) {
    $qid = (string)$posRow['questionId'];
    $bankId = (string)$posRow['bankId'];
    $qrow = $byId[$qid] ?? null;
    if (!$qrow) continue;

    $answerIndex = (int)$qrow['answerIndex'];
    $selectedIndex = array_key_exists($qid, $answers) ? (int)$answers[$qid] : -1;
    $isCorrect = ($selectedIndex === $answerIndex);
    if ($isCorrect) $correct++;

    $insAttempt->execute([
      'uid' => $me['id'],
      'bid' => $bankId,
      'qid' => $qid,
      'sel' => max(-1, $selectedIndex),
      'isc' => $isCorrect ? 1 : 0,
      'sec' => max(0, $per),
    ]);

    // wrong-queue update
    $uwqSel->execute(['uid' => $me['id'], 'bid' => $bankId, 'qid' => $qid]);
    $uwq = $uwqSel->fetch();

    if ($uwq) {
      $status = (string)($uwq['status'] ?? 'active');
      $reps = (int)($uwq['repetitions'] ?? 0);
      $interval = (int)($uwq['interval_days'] ?? 1);
      $ease = (float)($uwq['ease'] ?? 2.30);
      $wrongCount = (int)($uwq['wrong_count'] ?? 0);

      if ($isCorrect) {
        $reps += 1;
        if ($reps === 1) $interval = 1;
        else if ($reps === 2) $interval = 3;
        else $interval = max(1, (int)round($interval * $ease));
        $ease = min(2.50, $ease + 0.05);
        if ($reps >= 5) $status = 'mastered';
      } else {
        $wrongCount += 1;
        $reps = 0;
        $interval = 1;
        $ease = max(1.30, $ease - 0.20);
        $status = 'active';
      }

      $next = $now->modify('+' . $interval . ' days')->format('Y-m-d H:i:s');
      $uwqUpd->execute([
        'cid' => $courseId,
        'st' => $status,
        'r' => $reps,
        'i' => $interval,
        'e' => number_format($ease, 2, '.', ''),
        'w' => $wrongCount,
        'n' => $next,
        'uid' => $me['id'],
        'bid' => $bankId,
        'qid' => $qid,
      ]);
    } else {
      if (!$isCorrect) {
        $next = $now->modify('+1 days')->format('Y-m-d H:i:s');
        $uwqIns->execute([
          'uid' => $me['id'],
          'cid' => $courseId,
          'bid' => $bankId,
          'qid' => $qid,
          'n' => $next,
        ]);
      }
    }

    if (!$isCorrect) {
      $reviewWrong[] = [
        'questionId' => $qid,
        'bankId' => $bankId,
        'prompt' => (string)$qrow['prompt'],
        'options' => json_decode((string)$qrow['options'], true) ?: [],
        'selectedIndex' => $selectedIndex,
        'answerIndex' => $answerIndex,
        'explain' => (string)($qrow['explanationText'] ?? ''),
      ];
    }
  }

  $score = $total ? (int)round(($correct / $total) * 100) : 0;
  $attemptId = uuidv4();

  $pdo->prepare('
    INSERT INTO study_group_challenge_attempts (id, challenge_id, group_id, user_id, correct_count, total_questions, score_percent, seconds_total, submitted_at)
    VALUES (:id, :cid, :gid, :uid, :c, :t, :s, :sec, UTC_TIMESTAMP())
  ')->execute([
    'id' => $attemptId,
    'cid' => $challengeId,
    'gid' => $groupId,
    'uid' => $me['id'],
    'c' => $correct,
    't' => $total,
    's' => $score,
    'sec' => max(0, $secondsTotal),
  ]);

  // Update user_progress totals + streak (same logic as exam/submit)
  $p = $pdo->prepare('SELECT streak, total_answered, correct_answered, study_seconds, last_active FROM user_progress WHERE user_id = :uid LIMIT 1');
  $p->execute(['uid' => $me['id']]);
  $pr = $p->fetch();

  $today = today_iso_date();
  $yesterday = (new DateTimeImmutable($today, new DateTimeZone('UTC')))->modify('-1 day')->format('Y-m-d');

  $streak = 1;
  $totalAnswered = 0;
  $correctAnswered = 0;
  $studySeconds = 0;
  $lastActive = $today;

  if ($pr) {
    $streak = (int)$pr['streak'];
    $totalAnswered = (int)$pr['total_answered'];
    $correctAnswered = (int)$pr['correct_answered'];
    $studySeconds = (int)$pr['study_seconds'];
    $lastActive = (string)$pr['last_active'];
  } else {
    $pdo->prepare('INSERT INTO user_progress (user_id, streak, accuracy, total_answered, correct_answered, study_seconds, last_active)
                   VALUES (:uid, 1, 0, 0, 0, 0, :d)')
      ->execute(['uid' => $me['id'], 'd' => $today]);
  }

  if ($lastActive !== $today) {
    if ($lastActive === $yesterday) $streak = max(1, $streak + 1);
    else $streak = 1;
  }

  $totalAnswered += $total;
  $correctAnswered += $correct;
  $studySeconds += max(0, $secondsTotal);
  $accuracy = $totalAnswered ? (int)round(($correctAnswered / $totalAnswered) * 100) : 0;

  $pdo->prepare('UPDATE user_progress
    SET streak = :s, accuracy = :a, total_answered = :t, correct_answered = :c, study_seconds = :ss, last_active = :d
    WHERE user_id = :uid'
  )->execute([
    's' => $streak,
    'a' => $accuracy,
    't' => $totalAnswered,
    'c' => $correctAnswered,
    'ss' => $studySeconds,
    'd' => $today,
    'uid' => $me['id'],
  ]);

  // Gamification (best-effort) - small XP bonus for group participation
  $gOut = null;
  try {
    $g = $pdo->prepare('SELECT daily_goal, xp, level, badges_json FROM user_gamification WHERE user_id = :uid LIMIT 1');
    $g->execute([':uid' => $me['id']]);
    $gr = $g->fetch();

    $dailyGoal = 10;
    $xp = 0;
    $level = 1;
    $badges = [];
    if ($gr) {
      $dailyGoal = max(5, min(200, (int)($gr['daily_goal'] ?? 10)));
      $xp = (int)($gr['xp'] ?? 0);
      $level = max(1, (int)($gr['level'] ?? 1));
      $badges = json_decode((string)($gr['badges_json'] ?? '[]'), true) ?: [];
      if (!is_array($badges)) $badges = [];
    }

    $xpEarned = ($total * 8) + ($correct * 5) + 15;
    $xpNew = max(0, $xp + $xpEarned);
    $levelNew = (int)floor($xpNew / 250) + 1;
    $levelUp = $levelNew > $level;

    $tq = $pdo->prepare('SELECT COUNT(*) AS answered, SUM(CASE WHEN is_correct = 1 THEN 1 ELSE 0 END) AS correct
                        FROM user_question_attempts WHERE user_id = :uid AND DATE(created_at) = UTC_DATE()');
    $tq->execute(['uid' => $me['id']]);
    $tr = $tq->fetch() ?: [];
    $todayAnswered = (int)($tr['answered'] ?? 0);
    $todayCorrect = (int)($tr['correct'] ?? 0);

    $wants = [];
    if ($score >= 70) $wants[] = 'group_challenge_pass';
    if ($score >= 90) $wants[] = 'group_challenge_distinction';
    if ($todayAnswered >= $dailyGoal && $dailyGoal >= 10) $wants[] = 'goal_complete';

    $badgeSet = array_values(array_unique(array_filter(array_map('strval', $badges))));
    $newBadges = [];
    foreach ($wants as $k) {
      if (!in_array($k, $badgeSet, true)) {
        $badgeSet[] = $k;
        $newBadges[] = $k;
      }
    }

    if (!$gr) {
      $pdo->prepare('INSERT INTO user_gamification (user_id, daily_goal, xp, level, badges_json, updated_at)
                     VALUES (:uid, :g, :xp, :lvl, :b, UTC_TIMESTAMP())')
        ->execute([
          'uid' => $me['id'],
          'g' => $dailyGoal,
          'xp' => $xpNew,
          'lvl' => $levelNew,
          'b' => json_encode(array_values($badgeSet), JSON_UNESCAPED_UNICODE),
        ]);
    } else {
      $pdo->prepare('UPDATE user_gamification SET xp = :xp, level = :lvl, badges_json = :b, updated_at = UTC_TIMESTAMP() WHERE user_id = :uid')
        ->execute([
          'xp' => $xpNew,
          'lvl' => $levelNew,
          'b' => json_encode(array_values($badgeSet), JSON_UNESCAPED_UNICODE),
          'uid' => $me['id'],
        ]);
    }

    $gOut = [
      'dailyGoal' => $dailyGoal,
      'todayAnswered' => $todayAnswered,
      'todayCorrect' => $todayCorrect,
      'xp' => $xpNew,
      'level' => $levelNew,
      'xpEarned' => $xpEarned,
      'levelUp' => $levelUp,
      'badges' => array_values($badgeSet),
      'badgesNew' => $newBadges,
    ];
  } catch (Throwable $e) {
    $gOut = null;
  }

  $pdo->commit();

  json_response([
    'success' => true,
    'data' => [
      'challengeId' => $challengeId,
      'groupId' => $groupId,
      'scorePercent' => $score,
      'correct' => $correct,
      'total' => $total,
      'secondsTotal' => max(0, $secondsTotal),
      'reviewWrong' => $reviewWrong,
      'progress' => $gOut ? array_merge([
        'streak' => $streak,
        'accuracy' => $accuracy,
        'totalAnswered' => $totalAnswered,
        'correctAnswered' => $correctAnswered,
        'studySeconds' => $studySeconds,
        'lastActive' => $today,
      ], $gOut) : [
        'streak' => $streak,
        'accuracy' => $accuracy,
        'totalAnswered' => $totalAnswered,
        'correctAnswered' => $correctAnswered,
        'studySeconds' => $studySeconds,
        'lastActive' => $today,
      ],
    ]
  ]);
} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  json_response(['success' => false, 'error' => 'Failed to submit challenge'], 500);
}
