<?php
// public_html/api/config.php
declare(strict_types=1);

// 1) SUPPRESS ON-SCREEN ERRORS (Crucial for API/CORS)
ini_set('display_errors', '0');
ini_set('display_startup_errors', '0');
error_reporting(E_ALL); // Errors go to error_log, not screen.

// 2) USE REQUIRE_ONCE
require_once __DIR__ . '/response.php';

// 3) SAFE DEFINE HELPER
if (!function_exists('safe_define')) {
    function safe_define(string $name, $value): void {
        if (!defined($name)) {
            define($name, $value);
        }
    }
}

// --------------------------------------------------
// OPTIONAL LOCAL OVERRIDES (LOAD EARLY)
// --------------------------------------------------
// This must run BEFORE any safe_define('DB_*') so it can set env vars.
$local = __DIR__ . '/config.local.php';
if (is_file($local)) {
    require_once $local;
}

// --------------------------------------------------
// ENV / Config
// --------------------------------------------------

// App
safe_define('APP_NAME', getenv('JABUSTUDYHUB_APP_NAME') ?: 'JabuStudyHub');

// IMPORTANT: set this to your frontend origin (Vercel URL) in production.
// Example: https://your-frontend.vercel.app
safe_define('CORS_ORIGIN', getenv('JABUSPARK_CORS_ORIGIN') ?: '*');

// Database (MySQL)
// IMPORTANT: Do NOT hardcode secrets in source code.
// Set these in env vars or config.local.php.
safe_define('DB_HOST', getenv('JABUSPARK_DB_HOST') ?: 'localhost');
safe_define('DB_NAME', getenv('JABUSPARK_DB_NAME') ?: 'hrfjgoot_jabuspark');
safe_define('DB_USER', getenv('JABUSPARK_DB_USER') ?: 'hrfjgoot_Gratitude');
safe_define('DB_PASS', getenv('JABUSPARK_DB_PASS') ?: ''); // now correctly filled via config.local.php/env

// Sessions
safe_define('SESSION_TTL_DAYS', 30);

// Setup key
safe_define('SETUP_KEY', getenv('JABUSPARK_SETUP_KEY') ?: 'CHANGE_ME_SETUP_KEY');

// File uploads
safe_define('UPLOAD_DIR', __DIR__ . '/uploads');

/**
 * Force correct public URLs for files served by the backend.
 * Example: https://jabumarket.com.ng/api/uploads/...
 */
safe_define('PUBLIC_BASE', rtrim(getenv('JABUSPARK_PUBLIC_BASE') ?: 'https://jabumarket.com.ng/api', '/'));

// --------------------------------------------------
// Gemini AI (server-side ONLY)
// --------------------------------------------------
// Recommended env var options:
//
// Option A (best): GEMINI_API_KEYS="KEY1,KEY2"  (comma-separated)
// Option B: GEMINI_API_KEY_PRIMARY="KEY1" and GEMINI_API_KEY_SECONDARY="KEY2"
//
// DO NOT hardcode keys here.

$geminiKeys = [];

// Option A: single env var CSV list
$keysCsv = getenv('GEMINI_API_KEYS') ?: '';
if ($keysCsv !== '') {
    $geminiKeys = array_values(array_filter(array_map('trim', explode(',', $keysCsv))));
}

// Option B: two env vars
if (!$geminiKeys) {
    $k1 = trim(getenv('GEMINI_API_KEY_PRIMARY') ?: '');
    $k2 = trim(getenv('GEMINI_API_KEY_SECONDARY') ?: '');
    $geminiKeys = array_values(array_filter([$k1, $k2]));
}

// Define key pool constant (array)
safe_define('GEMINI_API_KEYS', $geminiKeys);

// Backward compatibility: first key as GEMINI_API_KEY
safe_define('GEMINI_API_KEY', (is_array(GEMINI_API_KEYS) && isset(GEMINI_API_KEYS[0])) ? GEMINI_API_KEYS[0] : '');

// Model
safe_define('GEMINI_MODEL', getenv('GEMINI_MODEL') ?: 'gemini-2.5-flash-lite');

// Helper: pick a key by attempt number (0,1,2...) for retry fallback
if (!function_exists('gemini_pick_key')) {
    function gemini_pick_key(int $attempt = 0): ?string {
        $keys = (defined('GEMINI_API_KEYS') && is_array(GEMINI_API_KEYS)) ? GEMINI_API_KEYS : [];
        if (!$keys) return null;
        return $keys[$attempt % count($keys)];
    }
}

// --------------------------------------------------
// PDO
// --------------------------------------------------
// Only connect if $pdo isn't already set
if (!isset($pdo)) {
    $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4';

    try {
        $pdo = new PDO($dsn, DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ]);
    } catch (Throwable $e) {
        // Don’t leak details to the browser
        if (function_exists('json_response')) {
            json_response(['success' => false, 'error' => 'Database connection failed'], 500);
        } else {
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => 'Database connection failed']);
            exit;
        }
    }
}
