<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/_helpers.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$body = read_json();
require_fields($body, ['bankId']);

$bankId = trim((string)$body['bankId']);
$questionCount = isset($body['questionCount']) ? (int)$body['questionCount'] : 10;
$durationMins = isset($body['durationMins']) ? (int)$body['durationMins'] : 10;

$questionCount = max(5, min(30, $questionCount));
$durationMins  = max(3, min(60, $durationMins));

/**
 * Generate UUID v4 (36-char) for tables that use CHAR(36) primary keys.
 */
// Bank sanity
$b = $pdo->prepare('SELECT id, title, course_id AS courseId FROM banks WHERE id = :id LIMIT 1');
$b->execute(['id' => $bankId]);
$bank = $b->fetch();
if (!$bank) json_response(['success' => false, 'error' => 'Practice bank not found'], 404);

// Pick questions from the bank WITHOUT ORDER BY RAND() (scales much better)
// Strategy: choose a random offset against a stable ordering.
// Note: This is not perfectly uniform random, but avoids the heavy sort that RAND() triggers.
$cntStmt = $pdo->prepare('SELECT COUNT(*) FROM questions WHERE bank_id = :bid');
$cntStmt->execute(['bid' => $bankId]);
$totalInBank = (int)$cntStmt->fetchColumn();

if ($totalInBank <= 0) {
  json_response(['success' => false, 'error' => 'Bank has no questions'], 422);
}

$take = max(2, min($questionCount, $totalInBank));
$maxOffset = max(0, $totalInBank - $take);
$offset = $maxOffset > 0 ? random_int(0, $maxOffset) : 0;

// ORDER BY id should be index-friendly on most schemas and keeps results deterministic.
$qq = $pdo->prepare('SELECT id FROM questions WHERE bank_id = :bid ORDER BY id ASC LIMIT ' . (int)$offset . ', ' . (int)$take);
$qq->execute(['bid' => $bankId]);
$rows = $qq->fetchAll() ?: [];

// Extremely small banks (or edge cases) could still yield fewer rows than requested.
// If so, top up from the start of the ordering.
if (count($rows) < $take) {
  $need = $take - count($rows);
  $qq2 = $pdo->prepare('SELECT id FROM questions WHERE bank_id = :bid ORDER BY id ASC LIMIT ' . (int)$need);
  $qq2->execute(['bid' => $bankId]);
  $rows = array_merge($rows, $qq2->fetchAll() ?: []);
}
if (count($rows) < 2) {
  json_response(['success' => false, 'error' => 'Bank has too few questions for a duel'], 422);
}
$qids = array_values(array_map(fn($r) => (string)$r['id'], $rows));

// Create duel
$duelId = duel_new_id();   // your duel_links.id (varchar(64) in related tables)
$code   = duel_new_code();

// Ensure code unique
for ($i = 0; $i < 5; $i++) {
  $c = $pdo->prepare('SELECT 1 FROM duel_links WHERE code = :c LIMIT 1');
  $c->execute(['c' => $code]);
  if (!$c->fetchColumn()) break;
  $code = duel_new_code();
}

try {
  $pdo->beginTransaction();

  // 1) duel_links (already includes id)
  $pdo->prepare('
    INSERT INTO duel_links (id, code, bank_id, created_by, status, question_count, duration_minutes, created_at, expires_at)
    VALUES (:id, :code, :bid, :uid, "waiting", :qc, :dm, UTC_TIMESTAMP(), DATE_ADD(UTC_TIMESTAMP(), INTERVAL 2 DAY))
  ')->execute([
    'id'   => $duelId,
    'code' => $code,
    'bid'  => $bankId,
    'uid'  => (string)$me['id'],
    'qc'   => count($qids),
    'dm'   => $durationMins,
  ]);

  // 2) duel_participants (DB: id CHAR(36) NOT NULL)
  $participantId = uuidv4();

  $pdo->prepare('
    INSERT INTO duel_participants (id, duel_id, user_id, slot)
    VALUES (:id, :did, :uid, 1)
  ')->execute([
    'id'  => $participantId,
    'did' => $duelId,
    'uid' => (string)$me['id'],
  ]);

  // 3) duel_link_questions (DB columns: id, duel_id, question_id, pos)
  $insQ = $pdo->prepare('
    INSERT INTO duel_link_questions (id, duel_id, question_id, pos)
    VALUES (:id, :did, :qid, :pos)
  ');

  $pos = 1;
  foreach ($qids as $qid) {
    $insQ->execute([
      'id'  => uuidv4(),
      'did' => $duelId,
      'qid' => $qid,
      'pos' => $pos,
    ]);
    $pos += 1;
  }

  $pdo->commit();
} catch (Throwable $e) {
  try { $pdo->rollBack(); } catch (Throwable $_) {}
  json_response(['success' => false, 'error' => 'Server error', 'detail' => $e->getMessage()], 500);
}

json_response([
  'success' => true,
  'data' => [
    'duel' => [
      'id' => $duelId,
      'code' => $code,
      'status' => 'waiting',
      'bankId' => $bankId,
      'bankTitle' => (string)($bank['title'] ?? ''),
      'courseId' => (string)($bank['courseId'] ?? ''),
      'questionCount' => count($qids),
      'durationMins' => $durationMins,
      'createdAt' => now_iso(),
      'expiresAt' => gmdate('c', time() + 2 * 86400),
      'sharePath' => '/duel/' . $code,
    ]
  ]
]);
