<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/_helpers.php';

// Side-effect endpoint: join a duel (and auto-start it once the 2nd player joins)
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$body = read_json();
$code = trim((string)($body['code'] ?? ''));
if ($code === '') json_response(['success' => false, 'error' => 'Missing code'], 422);

try {
  $pdo->beginTransaction();

  $stmt = $pdo->prepare('SELECT * FROM duel_links WHERE code = :c LIMIT 1 FOR UPDATE');
  $stmt->execute(['c' => $code]);
  $d = $stmt->fetch();
  if (!$d) {
    $pdo->rollBack();
    json_response(['success' => false, 'error' => 'Duel link not found'], 404);
  }

  $expiresAt = (string)($d['expires_at'] ?? '');
  if ($expiresAt !== '' && strtotime($expiresAt) !== false && strtotime($expiresAt) < time()) {
    $pdo->rollBack();
    json_response(['success' => false, 'error' => 'Duel link expired'], 410);
  }
  if ((string)($d['status'] ?? '') === 'completed') {
    $pdo->rollBack();
    json_response(['success' => false, 'error' => 'Duel already completed'], 409);
  }

  // Bank metadata (for UI)
  $bStmt = $pdo->prepare('SELECT title, course_id AS courseId FROM banks WHERE id = :id LIMIT 1');
  $bStmt->execute(['id' => (string)$d['bank_id']]);
  $bank = $bStmt->fetch() ?: [];

  // Check participant row
  $p = $pdo->prepare('SELECT slot, submitted_at, joined_at FROM duel_participants WHERE duel_id = :did AND user_id = :uid LIMIT 1');
  $p->execute(['did' => (string)$d['id'], 'uid' => (string)$me['id']]);
  $meP = $p->fetch();

  $countStmt = $pdo->prepare('SELECT COUNT(*) FROM duel_participants WHERE duel_id = :did');
  $countStmt->execute(['did' => (string)$d['id']]);
  $count = (int)$countStmt->fetchColumn();

  if (!$meP) {
    if ($count >= 2) {
      $pdo->rollBack();
      json_response(['success' => false, 'error' => 'This duel already has 2 players'], 409);
    }
    $slot = $count + 1;
    $pdo->prepare('INSERT INTO duel_participants (id, duel_id, user_id, slot, joined_at) VALUES (:id, :did, :uid, :slot, UTC_TIMESTAMP())')
        ->execute(['id' => uuidv4(), 'did' => (string)$d['id'], 'uid' => (string)$me['id'], 'slot' => $slot]);
    $count += 1;
    $meP = ['slot' => $slot, 'submitted_at' => null, 'joined_at' => gmdate('Y-m-d H:i:s')];
  }

  // Auto-start when 2nd player joins.
  $status = (string)($d['status'] ?? 'waiting');
  $startedAt = $d['started_at'] ? (string)$d['started_at'] : null;
  if ($status === 'waiting' && $count >= 2 && !$startedAt) {
    $pdo->prepare('UPDATE duel_links SET status = "live", started_at = UTC_TIMESTAMP() WHERE id = :id')
        ->execute(['id' => (string)$d['id']]);
    $status = 'live';
    $startedAt = gmdate('Y-m-d H:i:s');
  }

  // Participants list (for lobby)
  $partsStmt = $pdo->prepare('SELECT p.user_id AS userId, p.slot, p.score, p.seconds_total AS secondsTotal, p.submitted_at AS submittedAt,
                                     u.full_name AS fullName
                              FROM duel_participants p
                              JOIN users u ON u.id = p.user_id
                              WHERE p.duel_id = :did
                              ORDER BY p.slot ASC');
  $partsStmt->execute(['did' => (string)$d['id']]);
  $parts = $partsStmt->fetchAll() ?: [];

  $pdo->commit();

  $durationMins = (int)($d['duration_minutes'] ?? 10);
  json_response([
    'success' => true,
    'data' => [
      'duel' => [
        'id' => (string)$d['id'],
        'code' => (string)$d['code'],
        'status' => $status,
        'bankId' => (string)$d['bank_id'],
        'bankTitle' => (string)($bank['title'] ?? ''),
        'courseId' => (string)($bank['courseId'] ?? ''),
        'questionCount' => (int)($d['question_count'] ?? 0),
        'durationMins' => $durationMins,
        'startedAt' => $startedAt,
        'remainingSeconds' => duel_safe_next_seconds($durationMins, $startedAt),
        'sharePath' => '/duel/' . (string)$d['code'],
      ],
      'participants' => array_map(static function($r){
        return [
          'userId' => (string)$r['userId'],
          'fullName' => (string)($r['fullName'] ?? ''),
          'slot' => (int)($r['slot'] ?? 0),
          'score' => isset($r['score']) ? (int)$r['score'] : null,
          'secondsTotal' => isset($r['secondsTotal']) ? (int)$r['secondsTotal'] : null,
          'submittedAt' => $r['submittedAt'] ? (string)$r['submittedAt'] : null,
        ];
      }, $parts),
      'me' => [
        'slot' => (int)($meP['slot'] ?? 0),
        'submittedAt' => $meP['submitted_at'] ? (string)$meP['submitted_at'] : null,
      ]
    ]
  ]);

} catch (Throwable $e) {
  try { $pdo->rollBack(); } catch (Throwable $_) {}
  json_response(['success' => false, 'error' => 'Server error', 'detail' => $e->getMessage()], 500);
}
