<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/_helpers.php';

// Read-only endpoint: fetch duel questions once the duel is live.
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'GET') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$code = trim((string)($_GET['code'] ?? ''));
if ($code === '') json_response(['success' => false, 'error' => 'Missing code'], 422);

$dStmt = $pdo->prepare('
  SELECT d.id, d.code, d.bank_id AS bankId, b.title AS bankTitle, b.course_id AS courseId,
         d.status, d.question_count AS questionCount, d.duration_minutes AS durationMins,
         d.started_at AS startedAt, d.expires_at AS expiresAt
  FROM duel_links d
  JOIN banks b ON b.id = d.bank_id
  WHERE d.code = :c
  LIMIT 1
');
$dStmt->execute(['c' => $code]);
$d = $dStmt->fetch();
if (!$d) json_response(['success' => false, 'error' => 'Duel link not found'], 404);

$expiresAt = (string)($d['expiresAt'] ?? '');
if ($expiresAt !== '' && strtotime($expiresAt) !== false && strtotime($expiresAt) < time()) {
  json_response(['success' => false, 'error' => 'Duel link expired'], 410);
}

if ((string)($d['status'] ?? '') !== 'live' && (string)($d['status'] ?? '') !== 'completed') {
  json_response(['success' => false, 'error' => 'Duel has not started yet'], 409);
}

// Ensure caller is a participant
$pStmt = $pdo->prepare('SELECT 1 FROM duel_participants WHERE duel_id = :did AND user_id = :uid LIMIT 1');
$pStmt->execute(['did' => (string)$d['id'], 'uid' => (string)$me['id']]);
if (!$pStmt->fetchColumn()) {
  json_response(['success' => false, 'error' => 'You are not part of this duel'], 403);
}

$questions = [];
// Fetch ordered question IDs
$qrowsStmt = $pdo->prepare('SELECT question_id AS questionId, pos FROM duel_link_questions WHERE duel_id = :did ORDER BY pos ASC');
$qrowsStmt->execute(['did' => (string)$d['id']]);
$qrows = $qrowsStmt->fetchAll() ?: [];
$qids = array_map(fn($r) => (string)$r['questionId'], $qrows);

if (count($qids) > 0) {
  $in = implode(',', array_fill(0, count($qids), '?'));
  $qq = $pdo->prepare('SELECT id, bank_id AS bankId, prompt, options FROM questions WHERE id IN (' . $in . ')');
  $qq->execute($qids);
  $got = $qq->fetchAll() ?: [];
  $byId = [];
  foreach ($got as $r) $byId[(string)$r['id']] = $r;
  foreach ($qrows as $r) {
    $qid = (string)$r['questionId'];
    $q = $byId[$qid] ?? null;
    if (!$q) continue;
    $questions[] = [
      'id' => (string)$q['id'],
      'bankId' => (string)$q['bankId'],
      'prompt' => (string)$q['prompt'],
      'options' => json_decode((string)$q['options'], true) ?: [],
    ];
  }
}

$durationMins = (int)($d['durationMins'] ?? 10);
$startedAt = $d['startedAt'] ? (string)$d['startedAt'] : null;
json_response([
  'success' => true,
  'data' => [
    'duel' => [
      'id' => (string)$d['id'],
      'code' => (string)$d['code'],
      'status' => (string)$d['status'],
      'bankId' => (string)$d['bankId'],
      'bankTitle' => (string)($d['bankTitle'] ?? ''),
      'courseId' => (string)($d['courseId'] ?? ''),
      'questionCount' => (int)($d['questionCount'] ?? count($questions)),
      'durationMins' => $durationMins,
      'startedAt' => $startedAt,
      'remainingSeconds' => duel_safe_next_seconds($durationMins, $startedAt),
      'sharePath' => '/duel/' . (string)$d['code'],
    ],
    'questions' => $questions,
  ]
]);
