<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/_helpers.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'GET') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$code = trim((string)($_GET['code'] ?? ''));
if ($code === '') json_response(['success' => false, 'error' => 'Missing code'], 422);

$dStmt = $pdo->prepare('SELECT d.id, d.code, d.bank_id AS bankId, b.title AS bankTitle, d.status, d.question_count AS questionCount,
                                d.duration_minutes AS durationMins, d.started_at AS startedAt, d.completed_at AS completedAt
                         FROM duel_links d
                         JOIN banks b ON b.id = d.bank_id
                         WHERE d.code = :c
                         LIMIT 1');
$dStmt->execute(['c' => $code]);
$d = $dStmt->fetch();
if (!$d) json_response(['success' => false, 'error' => 'Duel link not found'], 404);

$pStmt = $pdo->prepare('SELECT p.user_id AS userId, p.slot, p.score, p.seconds_total AS secondsTotal, p.submitted_at AS submittedAt,
                               u.full_name AS fullName
                        FROM duel_participants p
                        JOIN users u ON u.id = p.user_id
                        WHERE p.duel_id = :did
                        ORDER BY p.slot ASC');
$pStmt->execute(['did' => (string)$d['id']]);
$parts = $pStmt->fetchAll() ?: [];

$winner = null;
if (count($parts) >= 2) {
  $a = $parts[0];
  $b = $parts[1];
  $as = (int)($a['score'] ?? -1);
  $bs = (int)($b['score'] ?? -1);
  if ($as > $bs) $winner = (string)$a['userId'];
  else if ($bs > $as) $winner = (string)$b['userId'];
  else {
    // tie-breaker: faster time (lower secondsTotal)
    $at = isset($a['secondsTotal']) ? (int)$a['secondsTotal'] : PHP_INT_MAX;
    $bt = isset($b['secondsTotal']) ? (int)$b['secondsTotal'] : PHP_INT_MAX;
    if ($at < $bt) $winner = (string)$a['userId'];
    else if ($bt < $at) $winner = (string)$b['userId'];
    else $winner = null;
  }
}

// Bonus metadata (idempotent, backed by duel_bonus_awards)
$bonusAwarded = null;
$winnerBonusXp = 50;
try {
  $m = $pdo->prepare('SELECT winner_user_id AS winnerUserIdSaved, winner_bonus_xp AS winnerBonusXp, bonus_awarded_at AS bonusAwardedAt
                      FROM duel_links WHERE id = :id LIMIT 1');
  $m->execute(['id' => (string)$d['id']]);
  $meta = $m->fetch();
  if ($meta) {
    if (!empty($meta['winnerUserIdSaved'])) {
      $winner = (string)$meta['winnerUserIdSaved'];
    }
    if (isset($meta['winnerBonusXp'])) {
      $winnerBonusXp = max(0, (int)$meta['winnerBonusXp']);
    }
    // Prefer duel_links timestamp if present
    if (!empty($meta['bonusAwardedAt'])) {
      $bonusAwarded = true;
    }
  }

  // If bonus_awarded_at isn't present/used, fall back to duel_bonus_awards existence.
  if ($bonusAwarded === null) {
    $ba = $pdo->prepare('SELECT 1 FROM duel_bonus_awards WHERE duel_id = :did LIMIT 1');
    $ba->execute(['did' => (string)$d['id']]);
    $bonusAwarded = (bool)$ba->fetchColumn();
  }
} catch (Throwable $_) {
  $bonusAwarded = null;
}

json_response([
  'success' => true,
  'data' => [
    'duel' => [
      'id' => (string)$d['id'],
      'code' => (string)$d['code'],
      'status' => (string)$d['status'],
      'bankId' => (string)$d['bankId'],
      'bankTitle' => (string)($d['bankTitle'] ?? ''),
      'questionCount' => (int)($d['questionCount'] ?? 0),
      'durationMins' => (int)($d['durationMins'] ?? 10),
      'startedAt' => $d['startedAt'] ? (string)$d['startedAt'] : null,
      'completedAt' => $d['completedAt'] ? (string)$d['completedAt'] : null,
      'winnerUserId' => $winner,
      'winnerBonusXp' => $winnerBonusXp,
      'bonusAwarded' => $bonusAwarded,
    ],
    'participants' => array_map(static function($r){
      return [
        'userId' => (string)$r['userId'],
        'fullName' => (string)($r['fullName'] ?? ''),
        'slot' => (int)($r['slot'] ?? 0),
        'score' => isset($r['score']) ? (int)$r['score'] : null,
        'secondsTotal' => isset($r['secondsTotal']) ? (int)$r['secondsTotal'] : null,
        'submittedAt' => $r['submittedAt'] ? (string)$r['submittedAt'] : null,
      ];
    }, $parts),
    'me' => [
      'userId' => (string)$me['id'],
    ],
  ]
]);
