<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/_helpers.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$body = read_json();
require_fields($body, ['code', 'answers']);

$code = trim((string)$body['code']);
$answers = is_array($body['answers']) ? $body['answers'] : [];
// Client-provided time is not trusted (users can manipulate timers).
// We'll compute seconds_total on the server based on the duel start / join time.
$clientSecondsTotal = isset($body['secondsTotal']) ? (int)$body['secondsTotal'] : 0;

if ($code === '') json_response(['success' => false, 'error' => 'Missing code'], 422);

try {
  $pdo->beginTransaction();

  $dStmt = $pdo->prepare('SELECT * FROM duel_links WHERE code = :c LIMIT 1 FOR UPDATE');
  $dStmt->execute(['c' => $code]);
  $d = $dStmt->fetch();
  if (!$d) {
    $pdo->rollBack();
    json_response(['success' => false, 'error' => 'Duel link not found'], 404);
  }

  if ((string)($d['status'] ?? '') !== 'live' && (string)($d['status'] ?? '') !== 'completed') {
    $pdo->rollBack();
    json_response(['success' => false, 'error' => 'Duel has not started yet'], 409);
  }

  $pStmt = $pdo->prepare('SELECT submitted_at, joined_at FROM duel_participants WHERE duel_id = :did AND user_id = :uid LIMIT 1');
  $pStmt->execute(['did' => (string)$d['id'], 'uid' => (string)$me['id']]);
  $p = $pStmt->fetch();
  if (!$p) {
    $pdo->rollBack();
    json_response(['success' => false, 'error' => 'You are not part of this duel'], 403);
  }
  if ($p['submitted_at']) {
    $pdo->rollBack();
    json_response(['success' => false, 'error' => 'Already submitted'], 409);
  }

  // Duel question ids (ordered)
  $qStmt = $pdo->prepare('SELECT question_id AS questionId FROM duel_link_questions WHERE duel_id = :did ORDER BY pos ASC');
  $qStmt->execute(['did' => (string)$d['id']]);
  $qRows = $qStmt->fetchAll() ?: [];
  $qids = array_values(array_map(fn($r) => (string)$r['questionId'], $qRows));
  if (count($qids) === 0) {
    $pdo->rollBack();
    json_response(['success' => false, 'error' => 'Duel has no questions'], 500);
  }

  $in = implode(',', array_fill(0, count($qids), '?'));
  $ansStmt = $pdo->prepare('SELECT id, answer_index AS answerIndex FROM questions WHERE id IN (' . $in . ')');
  $ansStmt->execute($qids);
  $rows = $ansStmt->fetchAll() ?: [];
  $byId = [];
  foreach ($rows as $r) $byId[(string)$r['id']] = (int)($r['answerIndex'] ?? 0);

  $correct = 0;
  $total = count($qids);
  foreach ($qids as $qid) {
    if (!array_key_exists($qid, $answers)) continue;
    $sel = (int)$answers[$qid];
    $ai = $byId[$qid] ?? null;
    if ($ai === null) continue;
    if ($sel === (int)$ai) $correct += 1;
  }

  // Compute server-side elapsed time (cap to duel duration)
  $durationMins = (int)($d['duration_minutes'] ?? 10);
  $capSeconds = max(60, $durationMins * 60);
  $startedAtStr = $d['started_at'] ? (string)$d['started_at'] : '';
  $joinedAtStr = (string)($p['joined_at'] ?? '');

  $startTs = 0;
  $sa = strtotime($startedAtStr);
  if ($sa !== false) $startTs = max($startTs, $sa);
  $ja = strtotime($joinedAtStr);
  if ($ja !== false) $startTs = max($startTs, $ja);
  if ($startTs <= 0) $startTs = time();

  $secondsTotal = min($capSeconds, max(0, time() - $startTs));

  // attempt record (lightweight)
  $attemptId = uuidv4();
  $pdo->prepare('INSERT INTO duel_attempts (id, duel_id, user_id, answers_json, correct_count, total_count, seconds_total, created_at)
                 VALUES (:id, :did, :uid, :a, :c, :t, :s, UTC_TIMESTAMP())')
      ->execute([
        'id' => $attemptId,
        'did' => (string)$d['id'],
        'uid' => (string)$me['id'],
        'a' => json_encode($answers, JSON_UNESCAPED_UNICODE),
        'c' => $correct,
        't' => $total,
        's' => max(0, $secondsTotal),
      ]);

  $pdo->prepare('UPDATE duel_participants SET score = :sc, seconds_total = :sec, submitted_at = UTC_TIMESTAMP() WHERE duel_id = :did AND user_id = :uid')
      ->execute(['sc' => $correct, 'sec' => max(0, $secondsTotal), 'did' => (string)$d['id'], 'uid' => (string)$me['id']]);

  // If both submitted -> complete duel
  $doneStmt = $pdo->prepare('SELECT COUNT(*) FROM duel_participants WHERE duel_id = :did AND submitted_at IS NOT NULL');
  $doneStmt->execute(['did' => (string)$d['id']]);
  $doneCount = (int)$doneStmt->fetchColumn();

  $status = (string)($d['status'] ?? 'live');
  $winnerUserId = null;
  $winnerBonusXp = 0;
  $bonusAwarded = false;
  if ($doneCount >= 2) {
    // Only run completion logic once.
    $alreadyCompleted = !empty($d['completed_at']);
    if (!$alreadyCompleted) {
      // Compute winner (score desc, tie-breaker: faster time)
      $partsStmt = $pdo->prepare('SELECT user_id AS userId, score, seconds_total AS secondsTotal FROM duel_participants WHERE duel_id = :did ORDER BY slot ASC');
      $partsStmt->execute(['did' => (string)$d['id']]);
      $parts = $partsStmt->fetchAll() ?: [];
      if (count($parts) >= 2) {
        $a = $parts[0];
        $b = $parts[1];
        $as = isset($a['score']) ? (int)$a['score'] : -1;
        $bs = isset($b['score']) ? (int)$b['score'] : -1;
        if ($as > $bs) $winnerUserId = (string)$a['userId'];
        else if ($bs > $as) $winnerUserId = (string)$b['userId'];
        else {
          $at = isset($a['secondsTotal']) ? (int)$a['secondsTotal'] : PHP_INT_MAX;
          $bt = isset($b['secondsTotal']) ? (int)$b['secondsTotal'] : PHP_INT_MAX;
          if ($at < $bt) $winnerUserId = (string)$a['userId'];
          else if ($bt < $at) $winnerUserId = (string)$b['userId'];
          else $winnerUserId = null;
        }
      }

      // Mark completed
      $pdo->prepare('UPDATE duel_links SET status = "completed", completed_at = UTC_TIMESTAMP() WHERE id = :id')
          ->execute(['id' => (string)$d['id']]);

      // Winner XP bonus (idempotent): duel_bonus_awards.uq_duel_bonus_duel guarantees "award once".
      if ($winnerUserId) {
        $winnerBonusXp = isset($d['winner_bonus_xp']) ? (int)$d['winner_bonus_xp'] : 50;
        $winnerBonusXp = max(0, $winnerBonusXp);

        // Save winner_user_id for convenience in /duel/result
        try {
          $pdo->prepare('UPDATE duel_links SET winner_user_id = :w WHERE id = :id')
              ->execute(['w' => $winnerUserId, 'id' => (string)$d['id']]);
        } catch (Throwable $_) {
          // ignore if column missing
        }

        // Insert award row only once (unique on duel_id)
        try {
          $awardStmt = $pdo->prepare('INSERT IGNORE INTO duel_bonus_awards (id, duel_id, winner_user_id, bonus_xp, awarded_at)
                                      VALUES (:id, :did, :uid, :xp, UTC_TIMESTAMP())');
          $awardStmt->execute([
            'id' => uuidv4(),
            'did' => (string)$d['id'],
            'uid' => $winnerUserId,
            'xp' => $winnerBonusXp,
          ]);
          $bonusAwarded = ((int)$awardStmt->rowCount()) === 1;

          if ($bonusAwarded) {
            // Mark timestamp on duel_links if the column exists (nice for admin/debug)
            try {
              $pdo->prepare('UPDATE duel_links SET bonus_awarded_at = UTC_TIMESTAMP() WHERE id = :id')
                  ->execute(['id' => (string)$d['id']]);
            } catch (Throwable $_) {
              // ignore if column missing
            }

            // Apply XP to winner
            try {
              $g = $pdo->prepare('SELECT xp, level FROM user_gamification WHERE user_id = :uid LIMIT 1');
              $g->execute(['uid' => $winnerUserId]);
              $gr = $g->fetch();
              $xp = (int)($gr['xp'] ?? 0);
              $xpNew = $xp + $winnerBonusXp;
              $lvlNew = (int)floor($xpNew / 250) + 1;
              if ($gr) {
                $pdo->prepare('UPDATE user_gamification SET xp = :xp, level = :lvl, updated_at = UTC_TIMESTAMP() WHERE user_id = :uid')
                    ->execute(['xp' => $xpNew, 'lvl' => $lvlNew, 'uid' => $winnerUserId]);
              } else {
                $pdo->prepare('INSERT INTO user_gamification (user_id, daily_goal, xp, level, badges_json, updated_at)
                               VALUES (:uid, 10, :xp, :lvl, "[]", UTC_TIMESTAMP())')
                    ->execute(['uid' => $winnerUserId, 'xp' => $xpNew, 'lvl' => $lvlNew]);
              }
            } catch (Throwable $_) {
              // ignore if gamification tables missing
            }
          }
        } catch (Throwable $_) {
          // duel_bonus_awards table might not exist if migrations weren't run.
          $bonusAwarded = false;
        }
      }
    }
    $status = 'completed';
  }

  // Award small XP for participation
  try {
    $rewardXp = max(0, $correct * 2);
    if ($rewardXp > 0) {
      $g = $pdo->prepare('SELECT xp, level, daily_goal, badges_json FROM user_gamification WHERE user_id = :uid LIMIT 1');
      $g->execute(['uid' => (string)$me['id']]);
      $gr = $g->fetch();
      $xp = (int)($gr['xp'] ?? 0);
      $lvl = (int)($gr['level'] ?? 1);
      $xpNew = $xp + $rewardXp;
      $lvlNew = (int)floor($xpNew / 250) + 1;
      if ($gr) {
        $pdo->prepare('UPDATE user_gamification SET xp = :xp, level = :lvl, updated_at = UTC_TIMESTAMP() WHERE user_id = :uid')
            ->execute(['xp' => $xpNew, 'lvl' => $lvlNew, 'uid' => (string)$me['id']]);
      } else {
        $pdo->prepare('INSERT INTO user_gamification (user_id, daily_goal, xp, level, badges_json, updated_at)
                       VALUES (:uid, 10, :xp, :lvl, "[]", UTC_TIMESTAMP())')
            ->execute(['uid' => (string)$me['id'], 'xp' => $xpNew, 'lvl' => $lvlNew]);
      }
    }
  } catch (Throwable $_) {
    // If gamification tables are missing, ignore.
  }

  $pdo->commit();

  json_response([
    'success' => true,
    'data' => [
      'score' => $correct,
      'total' => $total,
      'status' => $status,
      'rewards' => [
        'xp' => max(0, $correct * 2),
      ],
      'duel' => [
        'code' => $code,
        'winnerUserId' => $winnerUserId,
        'winnerBonusXp' => $winnerBonusXp,
        'bonusAwarded' => $bonusAwarded,
        'secondsTotal' => $secondsTotal,
        'clientSecondsTotal' => max(0, $clientSecondsTotal),
      ],
    ],
  ]);

} catch (Throwable $e) {
  try { $pdo->rollBack(); } catch (Throwable $_) {}
  json_response(['success' => false, 'error' => 'Server error', 'detail' => $e->getMessage()], 500);
}
