<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'GET') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);

$courseId = isset($_GET['courseId']) ? trim((string)$_GET['courseId']) : '';
$count = isset($_GET['count']) ? (int)$_GET['count'] : 40;
$duration = isset($_GET['durationMins']) ? (int)$_GET['durationMins'] : 60;

if ($courseId === '') json_response(['success' => false, 'error' => 'Missing courseId'], 422);
$count = max(5, min(120, $count));
$duration = max(5, min(240, $duration));

// Ensure course exists
$c = $pdo->prepare('SELECT id, code, title, level FROM courses WHERE id = :id LIMIT 1');
$c->execute(['id' => $courseId]);
$course = $c->fetch();
if (!$course) json_response(['success' => false, 'error' => 'Course not found'], 404);

// Pull random questions for the course (across banks)
// NOTE: ORDER BY RAND() is okay for small datasets. Your course banks are currently small.
$q = $pdo->prepare('
  SELECT q.id, q.bank_id AS bankId, q.prompt, q.options
  FROM questions q
  JOIN banks b ON b.id = q.bank_id
  WHERE b.course_id = :cid
  ORDER BY RAND()
  LIMIT ' . (int)$count . '
');
$q->execute(['cid' => $courseId]);
$qs = $q->fetchAll() ?: [];
if (count($qs) === 0) json_response(['success' => false, 'error' => 'No questions found for this course'], 404);

$examId = uuidv4();

// Create session + map questions (best-effort: if tables don't exist, return a stateless exam)
$persisted = true;
try {
  $pdo->beginTransaction();
  $pdo->prepare('
    INSERT INTO exam_sessions (id, user_id, course_id, total_questions, duration_minutes, started_at)
    VALUES (:id, :uid, :cid, :t, :d, UTC_TIMESTAMP())
  ')->execute([
    'id' => $examId,
    'uid' => $me['id'],
    'cid' => $courseId,
    't' => count($qs),
    'd' => $duration,
  ]);

  $ins = $pdo->prepare('
    INSERT INTO exam_session_questions (exam_id, question_id, bank_id, position)
    VALUES (:eid, :qid, :bid, :pos)
  ');
  foreach ($qs as $i => $row) {
    $ins->execute([
      'eid' => $examId,
      'qid' => (string)$row['id'],
      'bid' => (string)$row['bankId'],
      'pos' => $i,
    ]);
  }
  $pdo->commit();
} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  $persisted = false;
}

// Hide answers
$questions = [];
foreach ($qs as $row) {
  $questions[] = [
    'id' => (string)$row['id'],
    'bankId' => (string)$row['bankId'],
    'prompt' => (string)$row['prompt'],
    'options' => json_decode((string)$row['options'], true) ?: [],
  ];
}

json_response([
  'success' => true,
  'data' => [
    'persisted' => $persisted,
    'exam' => [
      'id' => $examId,
      'courseId' => (string)$courseId,
      'courseCode' => (string)($course['code'] ?? $courseId),
      'courseTitle' => (string)($course['title'] ?? ''),
      'durationMins' => $duration,
      'total' => count($questions),
      'startedAt' => now_iso(),
    ],
    'questions' => $questions,
  ],
]);
