<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$body = read_json();
require_fields($body, ['examId', 'answers']);

$examId = (string)$body['examId'];
$answers = $body['answers']; // { [questionId]: selectedIndex }
$secondsTotal = isset($body['secondsTotal']) ? (int)$body['secondsTotal'] : 0;

if (!is_array($answers)) {
  json_response(['success' => false, 'error' => 'answers must be an object/map'], 422);
}

// Load exam session (if persisted)
$session = null;
try {
  $s = $pdo->prepare('SELECT * FROM exam_sessions WHERE id = :id AND user_id = :uid LIMIT 1');
  $s->execute(['id' => $examId, 'uid' => $me['id']]);
  $session = $s->fetch() ?: null;
} catch (Throwable $e) {
  $session = null;
}

if (!$session) {
  json_response([
    'success' => false,
    'error' => 'Exam session not found. Did you run the exam migration?',
  ], 404);
}

if (!empty($session['submitted_at'])) {
  json_response(['success' => false, 'error' => 'This exam was already submitted'], 409);
}

$courseId = (string)($session['course_id'] ?? '');

// Load exam questions (authoritative list)
$qq = $pdo->prepare('SELECT question_id AS questionId, bank_id AS bankId, position FROM exam_session_questions WHERE exam_id = :id ORDER BY position ASC');
$qq->execute(['id' => $examId]);
$examQuestions = $qq->fetchAll() ?: [];
if (count($examQuestions) === 0) {
  json_response(['success' => false, 'error' => 'Exam has no questions'], 500);
}

$qids = array_map(fn($r) => (string)$r['questionId'], $examQuestions);

// Fetch correct answers + explanations
$in = implode(',', array_fill(0, count($qids), '?'));
$stmt = $pdo->prepare(
  'SELECT q.id, q.bank_id AS bankId, q.prompt, q.options, q.answer_index AS answerIndex, q.explanation AS explanationText
   FROM questions q
   WHERE q.id IN (' . $in . ')'
);
$stmt->execute($qids);
$rows = $stmt->fetchAll() ?: [];

$byId = [];
foreach ($rows as $r) {
  $byId[(string)$r['id']] = $r;
}

$total = count($examQuestions);
$correct = 0;
$review = []; // keep wrong ones (and optionally a few correct)
$dueReviewClears = 0;

// Prepare inserts
$insAttempt = $pdo->prepare('
  INSERT INTO user_question_attempts (user_id, bank_id, question_id, selected_index, is_correct, seconds_spent, created_at)
  VALUES (:uid, :bid, :qid, :sel, :isc, :sec, UTC_TIMESTAMP())
');

// Wrong-questions queue helpers (same logic as practice/submit but simplified)
$uwqSel = $pdo->prepare('SELECT status, repetitions, interval_days, ease, wrong_count, next_review_at FROM user_wrong_questions WHERE user_id = :uid AND bank_id = :bid AND question_id = :qid LIMIT 1');
$uwqIns = $pdo->prepare('
  INSERT INTO user_wrong_questions
    (user_id, course_id, bank_id, question_id, status, repetitions, interval_days, ease, wrong_count, last_review_at, next_review_at, created_at, updated_at)
  VALUES
    (:uid, :cid, :bid, :qid, "active", 0, 1, 2.30, 1, UTC_TIMESTAMP(), :n, UTC_TIMESTAMP(), UTC_TIMESTAMP())
');
$uwqUpd = $pdo->prepare('
  UPDATE user_wrong_questions
  SET course_id = :cid,
      status = :st,
      repetitions = :r,
      interval_days = :i,
      ease = :e,
      wrong_count = :w,
      last_review_at = UTC_TIMESTAMP(),
      next_review_at = :n,
      updated_at = UTC_TIMESTAMP()
  WHERE user_id = :uid AND bank_id = :bid AND question_id = :qid
');

$now = new DateTimeImmutable('now', new DateTimeZone('UTC'));

try {
  $pdo->beginTransaction();

  // Evaluate each question
  foreach ($examQuestions as $posRow) {
    $qid = (string)$posRow['questionId'];
    $bankId = (string)$posRow['bankId'];
    $qrow = $byId[$qid] ?? null;
    if (!$qrow) continue;

    $answerIndex = (int)$qrow['answerIndex'];
    $selectedIndex = array_key_exists($qid, $answers) ? (int)$answers[$qid] : -1;
    $isCorrect = ($selectedIndex === $answerIndex);
    if ($isCorrect) $correct++;

    // log attempt (seconds_spent is distributed evenly if provided)
    $per = $secondsTotal > 0 ? (int)floor($secondsTotal / max(1, $total)) : 0;
    $insAttempt->execute([
      'uid' => $me['id'],
      'bid' => $bankId,
      'qid' => $qid,
      'sel' => max(-1, $selectedIndex),
      'isc' => $isCorrect ? 1 : 0,
      'sec' => max(0, $per),
    ]);

    // Update wrong-questions queue
    $uwqSel->execute(['uid' => $me['id'], 'bid' => $bankId, 'qid' => $qid]);
    $uwq = $uwqSel->fetch();

    // Was this question due for review at the time of attempt?
    $wasDueReview = false;
    try {
      if ($uwq) {
        $st = (string)($uwq['status'] ?? 'active');
        $nra = (string)($uwq['next_review_at'] ?? '');
        if ($st === 'active' && $nra !== '' && strtotime($nra) !== false && strtotime($nra) <= $now->getTimestamp()) {
          $wasDueReview = true;
        }
      }
    } catch (Throwable $e) {
      $wasDueReview = false;
    }

    if ($wasDueReview && $isCorrect) {
      $dueReviewClears += 1;
    }

    if ($uwq) {
      $status = (string)($uwq['status'] ?? 'active');
      $reps = (int)($uwq['repetitions'] ?? 0);
      $interval = (int)($uwq['interval_days'] ?? 1);
      $ease = (float)($uwq['ease'] ?? 2.30);
      $wrongCount = (int)($uwq['wrong_count'] ?? 0);

      if ($isCorrect) {
        $reps += 1;
        if ($reps === 1) $interval = 1;
        else if ($reps === 2) $interval = 3;
        else $interval = max(1, (int)round($interval * $ease));
        $ease = min(2.50, $ease + 0.05);
        if ($reps >= 5) $status = 'mastered';
      } else {
        $wrongCount += 1;
        $reps = 0;
        $interval = 1;
        $ease = max(1.30, $ease - 0.20);
        $status = 'active';
      }

      $next = $now->modify('+' . $interval . ' days')->format('Y-m-d H:i:s');
      $uwqUpd->execute([
        'cid' => $courseId,
        'st' => $status,
        'r' => $reps,
        'i' => $interval,
        'e' => number_format($ease, 2, '.', ''),
        'w' => $wrongCount,
        'n' => $next,
        'uid' => $me['id'],
        'bid' => $bankId,
        'qid' => $qid,
      ]);
    } else {
      if (!$isCorrect) {
        $next = $now->modify('+1 days')->format('Y-m-d H:i:s');
        $uwqIns->execute([
          'uid' => $me['id'],
          'cid' => $courseId,
          'bid' => $bankId,
          'qid' => $qid,
          'n' => $next,
        ]);
      }
    }

    if (!$isCorrect) {
      $review[] = [
        'questionId' => $qid,
        'bankId' => $bankId,
        'prompt' => (string)$qrow['prompt'],
        'options' => json_decode((string)$qrow['options'], true) ?: [],
        'selectedIndex' => $selectedIndex,
        'answerIndex' => $answerIndex,
        'explain' => (string)($qrow['explanationText'] ?? ''),
      ];
    }
  }

  $score = $total ? (int)round(($correct / $total) * 100) : 0;

  // Mark session submitted
  $pdo->prepare('UPDATE exam_sessions SET submitted_at = UTC_TIMESTAMP(), correct_count = :c, score_percent = :s, seconds_total = :sec WHERE id = :id')
    ->execute(['c' => $correct, 's' => $score, 'sec' => max(0, $secondsTotal), 'id' => $examId]);

  // Update user_progress totals + streak (same logic as practice)
  $pr = null;
  $streakFreezes = 0;
  try {
    $p = $pdo->prepare('SELECT streak, total_answered, correct_answered, study_seconds, last_active, streak_freezes FROM user_progress WHERE user_id = :uid LIMIT 1');
    $p->execute(['uid' => $me['id']]);
    $pr = $p->fetch();
    $streakFreezes = (int)($pr['streak_freezes'] ?? 0);
  } catch (Throwable $e) {
    $p = $pdo->prepare('SELECT streak, total_answered, correct_answered, study_seconds, last_active FROM user_progress WHERE user_id = :uid LIMIT 1');
    $p->execute(['uid' => $me['id']]);
    $pr = $p->fetch();
    $streakFreezes = 0;
  }

  $today = today_iso_date();
  $yesterday = (new DateTimeImmutable($today, new DateTimeZone('UTC')))->modify('-1 day')->format('Y-m-d');

  $streak = 1;
  $totalAnswered = 0;
  $correctAnswered = 0;
  $studySeconds = 0;
  $lastActive = $today;

  if ($pr) {
    $streak = (int)$pr['streak'];
    $totalAnswered = (int)$pr['total_answered'];
    $correctAnswered = (int)$pr['correct_answered'];
    $studySeconds = (int)$pr['study_seconds'];
    $lastActive = (string)$pr['last_active'];
  } else {
    try {
      $pdo->prepare('INSERT INTO user_progress (user_id, streak, accuracy, total_answered, correct_answered, study_seconds, last_active, streak_freezes)
                     VALUES (:uid, 1, 0, 0, 0, 0, :d, 0)')
        ->execute(['uid' => $me['id'], 'd' => $today]);
    } catch (Throwable $e) {
      $pdo->prepare('INSERT INTO user_progress (user_id, streak, accuracy, total_answered, correct_answered, study_seconds, last_active)
                     VALUES (:uid, 1, 0, 0, 0, 0, :d)')
        ->execute(['uid' => $me['id'], 'd' => $today]);
    }
  }

  if ($lastActive !== $today) {
    if ($lastActive === $yesterday) {
      $streak = max(1, $streak + 1);
    } else {
      // Streak freeze: if you missed exactly 1 day (gap=2) and have a freeze, keep streak.
      try {
        $dLast = new DateTimeImmutable($lastActive, new DateTimeZone('UTC'));
        $dToday = new DateTimeImmutable($today, new DateTimeZone('UTC'));
        $gap = (int)$dLast->diff($dToday)->format('%a');
        if ($gap === 2 && $streakFreezes > 0) {
          $streakFreezes = max(0, $streakFreezes - 1);
          $streak = max(1, $streak + 1);
        } else {
          $streak = 1;
        }
      } catch (Throwable $e) {
        $streak = 1;
      }
    }
  }

  $totalAnswered += $total;
  $correctAnswered += $correct;
  $studySeconds += max(0, $secondsTotal);
  $accuracy = $totalAnswered ? (int)round(($correctAnswered / $totalAnswered) * 100) : 0;

  try {
    $pdo->prepare('UPDATE user_progress
      SET streak = :s, accuracy = :a, total_answered = :t, correct_answered = :c, study_seconds = :ss, last_active = :d, streak_freezes = :sf
      WHERE user_id = :uid'
    )->execute([
      's' => $streak,
      'a' => $accuracy,
      't' => $totalAnswered,
      'c' => $correctAnswered,
      'ss' => $studySeconds,
      'd' => $today,
      'sf' => $streakFreezes,
      'uid' => $me['id'],
    ]);
  } catch (Throwable $e) {
    $pdo->prepare('UPDATE user_progress
      SET streak = :s, accuracy = :a, total_answered = :t, correct_answered = :c, study_seconds = :ss, last_active = :d
      WHERE user_id = :uid'
    )->execute([
      's' => $streak,
      'a' => $accuracy,
      't' => $totalAnswered,
      'c' => $correctAnswered,
      'ss' => $studySeconds,
      'd' => $today,
      'uid' => $me['id'],
    ]);
  }

  // Weekly Missions (best-effort)
  $missionsClaimable = null;
  try {
    $missionsHelper = __DIR__ . '/../gamification/_missions.php';
    if (is_file($missionsHelper)) {
      require_once $missionsHelper;
      $weekStart = missions_week_start_utc();
      missions_increment($pdo, (string)$me['id'], [
        'answered' => $total,
        'correct' => $correct,
        'studySeconds' => max(0, $secondsTotal),
        'dueReviews' => $dueReviewClears,
      ], $weekStart);
      $missionsClaimable = missions_claimable_count($pdo, (string)$me['id'], $weekStart);
    }
  } catch (Throwable $e) {
    $missionsClaimable = null;
  }

  // Gamification: award XP and badges (best-effort)
  $gOut = null;
  try {
    $g = $pdo->prepare('SELECT daily_goal, xp, level, badges_json FROM user_gamification WHERE user_id = :uid LIMIT 1');
    $g->execute([':uid' => $me['id']]);
    $gr = $g->fetch();

    $dailyGoal = 10;
    $xp = 0;
    $level = 1;
    $badges = [];
    if ($gr) {
      $dailyGoal = max(5, min(200, (int)($gr['daily_goal'] ?? 10)));
      $xp = (int)($gr['xp'] ?? 0);
      $level = max(1, (int)($gr['level'] ?? 1));
      $badges = json_decode((string)($gr['badges_json'] ?? '[]'), true) ?: [];
      if (!is_array($badges)) $badges = [];
    }

    $xpEarned = ($total * 8) + ($correct * 5);
    $xpNew = max(0, $xp + $xpEarned);
    $levelNew = (int)floor($xpNew / 250) + 1;
    $levelUp = $levelNew > $level;

    // Today counts
    $t = $pdo->prepare('SELECT COUNT(*) AS answered, SUM(CASE WHEN is_correct = 1 THEN 1 ELSE 0 END) AS correct
                        FROM user_question_attempts WHERE user_id = :uid AND DATE(created_at) = UTC_DATE()');
    $t->execute(['uid' => $me['id']]);
    $tr = $t->fetch() ?: [];
    $todayAnswered = (int)($tr['answered'] ?? 0);
    $todayCorrect = (int)($tr['correct'] ?? 0);

    $wants = [];
    if ($total >= 20) $wants[] = 'exam_20';
    if ($score >= 70) $wants[] = 'exam_pass';
    if ($score >= 90) $wants[] = 'exam_distinction';
    if ($streak >= 7) $wants[] = 'streak_7';
    if ($todayAnswered >= $dailyGoal && $dailyGoal >= 10) $wants[] = 'goal_complete';

    $badgeSet = array_values(array_unique(array_filter(array_map('strval', $badges))));
    $newBadges = [];
    foreach ($wants as $k) {
      if (!in_array($k, $badgeSet, true)) {
        $badgeSet[] = $k;
        $newBadges[] = $k;
      }
    }

    // Upsert row if missing
    if (!$gr) {
      $pdo->prepare('INSERT INTO user_gamification (user_id, daily_goal, xp, level, badges_json, updated_at)
                     VALUES (:uid, :g, :xp, :lvl, :b, UTC_TIMESTAMP())')
        ->execute([
          'uid' => $me['id'],
          'g' => $dailyGoal,
          'xp' => $xpNew,
          'lvl' => $levelNew,
          'b' => json_encode(array_values($badgeSet), JSON_UNESCAPED_UNICODE),
        ]);
    } else {
      $pdo->prepare('UPDATE user_gamification SET xp = :xp, level = :lvl, badges_json = :b, updated_at = UTC_TIMESTAMP() WHERE user_id = :uid')
        ->execute([
          'xp' => $xpNew,
          'lvl' => $levelNew,
          'b' => json_encode(array_values($badgeSet), JSON_UNESCAPED_UNICODE),
          'uid' => $me['id'],
        ]);
    }

    $gOut = [
      'dailyGoal' => $dailyGoal,
      'todayAnswered' => $todayAnswered,
      'todayCorrect' => $todayCorrect,
      'xp' => $xpNew,
      'level' => $levelNew,
      'xpEarned' => $xpEarned,
      'levelUp' => $levelUp,
      'badges' => array_values($badgeSet),
      'badgesNew' => $newBadges,
    ];
  } catch (Throwable $e) {
    $gOut = null;
  }

  $pdo->commit();

  json_response([
    'success' => true,
    'data' => [
      'examId' => $examId,
      'courseId' => $courseId,
      'total' => $total,
      'correct' => $correct,
      'scorePercent' => $score,
      'secondsTotal' => max(0, $secondsTotal),
      'reviewWrong' => $review,
      'progress' => array_merge([
        'streak' => $streak,
        'streakFreezes' => $streakFreezes,
        'accuracy' => $accuracy,
        'totalAnswered' => $totalAnswered,
        'correctAnswered' => $correctAnswered,
        'studySeconds' => $studySeconds,
        'lastActive' => $today,
        'missionsClaimable' => $missionsClaimable,
      ], is_array($gOut) ? $gOut : []),
    ]
  ]);
} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  json_response(['success' => false, 'error' => 'Failed to submit exam'], 500);
}
