<?php
declare(strict_types=1);

/**
 * Weekly Missions + Streak Freeze helpers.
 *
 * Safety model:
 * - Missions are "optional". If tables/columns aren't present yet, callers may catch exceptions.
 * - This helper also includes a lightweight "sync" from user_question_attempts so missions are
 *   still accurate even if the feature was deployed mid-week (or if an earlier bug prevented
 *   increment updates).
 */

function missions_week_start_utc(): string {
  $now = new DateTimeImmutable('now', new DateTimeZone('UTC'));
  $dow = (int)$now->format('N'); // 1..7 (Mon..Sun)
  return $now->modify('-' . ($dow - 1) . ' days')->format('Y-m-d');
}

function missions_week_end_utc(string $weekStart): string {
  $start = new DateTimeImmutable($weekStart . ' 00:00:00', new DateTimeZone('UTC'));
  return $start->modify('+7 days')->format('Y-m-d'); // exclusive end date
}

/**
 * Seeds default missions for a given user/week.
 */
function missions_seed_if_missing(PDO $pdo, string $userId, string $weekStart): void {
  $chk = $pdo->prepare('SELECT 1 FROM weekly_missions WHERE user_id = :uid AND week_start = :w LIMIT 1');
  $chk->execute(['uid' => $userId, 'w' => $weekStart]);
  if ($chk->fetch()) return;

  $missions = [
    [
      'key' => 'weekly_questions',
      'title' => 'Answer 60 questions',
      'description' => 'Build momentum: hit 60 attempts this week.',
      'goal' => 60,
      'xp' => 80,
      'freezes' => 0,
    ],
    [
      'key' => 'weekly_correct',
      'title' => 'Get 30 correct',
      'description' => 'Accuracy matters: get 30 correct answers.',
      'goal' => 30,
      'xp' => 70,
      'freezes' => 0,
    ],
    [
      'key' => 'weekly_study_seconds',
      'title' => 'Study 30 minutes',
      'description' => 'Spend 30 minutes in practice/exam mode.',
      'goal' => 1800,
      'xp' => 60,
      'freezes' => 0,
    ],
    [
      'key' => 'weekly_due_reviews',
      'title' => 'Clear 10 due reviews',
      'description' => 'Knock out 10 due Smart Review items.',
      'goal' => 10,
      'xp' => 70,
      'freezes' => 1,
    ],
  ];

  // DB column is "goal" (frontend uses "target"; we alias on read)
  $ins = $pdo->prepare('INSERT INTO weekly_missions
    (user_id, week_start, mission_key, title, description, progress, goal, reward_xp, reward_freezes, completed_at, claimed_at)
    VALUES
    (:uid, :w, :k, :t, :d, 0, :goal, :xp, :fz, NULL, NULL)'
  );

  foreach ($missions as $m) {
    $ins->execute([
      'uid' => $userId,
      'w' => $weekStart,
      'k' => $m['key'],
      't' => $m['title'],
      'd' => $m['description'],
      'goal' => (int)$m['goal'],
      'xp' => (int)$m['xp'],
      'fz' => (int)$m['freezes'],
    ]);
  }
}

/**
 * Sync missions from existing activity logs.
 *
 * Why this exists:
 * - If you deploy weekly_missions mid-week, users may have already answered a lot of questions.
 * - If missions_increment was failing silently (try/catch in submit endpoints), progress stays 0.
 *
 * This recomputes 3 missions from user_question_attempts for the given week:
 * - weekly_questions      (COUNT(*))
 * - weekly_correct        (SUM(is_correct=1))
 * - weekly_study_seconds  (SUM(seconds_spent))
 *
 * Note: weekly_due_reviews is NOT reliably backfillable because "was due review" is not stored in
 * user_question_attempts.
 */
function missions_sync_from_attempts(PDO $pdo, string $userId, string $weekStart): void {
  try {
    missions_seed_if_missing($pdo, $userId, $weekStart);

    $startTs = $weekStart . ' 00:00:00';
    $endDate = missions_week_end_utc($weekStart);
    $endTs = $endDate . ' 00:00:00';

    $q = $pdo->prepare('SELECT
                          COUNT(*) AS answered,
                          COALESCE(SUM(CASE WHEN is_correct = 1 THEN 1 ELSE 0 END), 0) AS correct,
                          COALESCE(SUM(seconds_spent), 0) AS studySeconds
                        FROM user_question_attempts
                        WHERE user_id = :uid
                          AND created_at >= :start
                          AND created_at < :end');
    $q->execute([
      'uid' => $userId,
      'start' => $startTs,
      'end' => $endTs,
    ]);
    $agg = $q->fetch() ?: [];

    $answered = (int)($agg['answered'] ?? 0);
    $correct = (int)($agg['correct'] ?? 0);
    $studySeconds = (int)($agg['studySeconds'] ?? 0);

    $updates = [
      'weekly_questions' => $answered,
      'weekly_correct' => $correct,
      'weekly_study_seconds' => $studySeconds,
    ];

    $upd = $pdo->prepare('UPDATE weekly_missions
                          SET progress = LEAST(goal, :p),
                              completed_at = CASE
                                WHEN completed_at IS NULL AND :p >= goal THEN UTC_TIMESTAMP()
                                ELSE completed_at
                              END
                          WHERE user_id = :uid AND week_start = :w AND mission_key = :k');

    foreach ($updates as $key => $p) {
      $upd->execute([
        'p' => max(0, (int)$p),
        'uid' => $userId,
        'w' => $weekStart,
        'k' => $key,
      ]);
    }
  } catch (Throwable $e) {
    // Missions must never break core flows.
    // We intentionally swallow errors here.
  }
}

/**
 * Increment mission progress in a single request.
 *
 * $deltas keys:
 * - answered (int)
 * - correct (int)
 * - studySeconds (int)
 * - dueReviews (int)
 */
function missions_increment(PDO $pdo, string $userId, array $deltas, string $weekStart): void {
  missions_seed_if_missing($pdo, $userId, $weekStart);

  $map = [
    'answered' => 'weekly_questions',
    'correct' => 'weekly_correct',
    'studySeconds' => 'weekly_study_seconds',
    'dueReviews' => 'weekly_due_reviews',
  ];

  foreach ($map as $deltaKey => $missionKey) {
    $inc = (int)($deltas[$deltaKey] ?? 0);
    if ($inc <= 0) continue;

    $pdo->prepare('
      UPDATE weekly_missions
      SET progress = LEAST(goal, progress + :inc),
          completed_at = CASE
            WHEN completed_at IS NULL AND (progress + :inc) >= goal THEN UTC_TIMESTAMP()
            ELSE completed_at
          END
      WHERE user_id = :uid AND week_start = :w AND mission_key = :k
    ')->execute([
      'inc' => $inc,
      'uid' => $userId,
      'w' => $weekStart,
      'k' => $missionKey,
    ]);
  }
}

function missions_list(PDO $pdo, string $userId, string $weekStart): array {
  missions_seed_if_missing($pdo, $userId, $weekStart);
  // Backfill/sync from attempts (safe; swallowed on error)
  missions_sync_from_attempts($pdo, $userId, $weekStart);

  // Select goal AS target so frontend can keep using "target"
  $q = $pdo->prepare('SELECT mission_key AS missionKey, title, description, progress, goal AS target,
                             reward_xp AS rewardXp, reward_freezes AS rewardFreezes,
                             completed_at AS completedAt, claimed_at AS claimedAt
                      FROM weekly_missions
                      WHERE user_id = :uid AND week_start = :w
                      ORDER BY mission_key ASC');
  $q->execute(['uid' => $userId, 'w' => $weekStart]);
  $rows = $q->fetchAll() ?: [];

  $out = [];
  foreach ($rows as $r) {
    $progress = (int)($r['progress'] ?? 0);
    $target = (int)($r['target'] ?? 0);

    $out[] = [
      'missionKey' => (string)$r['missionKey'],
      'title' => (string)$r['title'],
      'description' => (string)$r['description'],
      'progress' => $progress,
      'target' => $target,
      'rewardXp' => (int)($r['rewardXp'] ?? 0),
      'rewardFreezes' => (int)($r['rewardFreezes'] ?? 0),
      'isComplete' => $target > 0 ? ($progress >= $target) : false,
      'isClaimed' => !empty($r['claimedAt']),
      'completedAt' => $r['completedAt'] ? (string)$r['completedAt'] : null,
      'claimedAt' => $r['claimedAt'] ? (string)$r['claimedAt'] : null,
    ];
  }

  return $out;
}

function missions_claimable_count(PDO $pdo, string $userId, string $weekStart): int {
  // Keep DB in sync before counting.
  missions_sync_from_attempts($pdo, $userId, $weekStart);

  $q = $pdo->prepare('SELECT COUNT(*) AS c
                      FROM weekly_missions
                      WHERE user_id = :uid AND week_start = :w
                        AND progress >= goal AND claimed_at IS NULL');
  $q->execute(['uid' => $userId, 'w' => $weekStart]);
  return (int)(($q->fetch() ?: [])['c'] ?? 0);
}
