<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/_missions.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$body = read_json();
require_fields($body, ['missionKey']);

$missionKey = trim((string)$body['missionKey']);
if ($missionKey === '') {
  json_response(['success' => false, 'error' => 'missionKey is required'], 422);
}

try {
  $weekStart = missions_week_start_utc();

  $pdo->beginTransaction();

  // Ensure week exists
  missions_seed_if_missing($pdo, (string)$me['id'], $weekStart);

  // Sync from attempts so missions can be claimable even if deployed mid-week
  missions_sync_from_attempts($pdo, (string)$me['id'], $weekStart);

  // ✅ DB column is goal, alias it to target for compatibility
  $rowStmt = $pdo->prepare('SELECT mission_key, progress, goal AS target, reward_xp, reward_freezes, claimed_at
                            FROM weekly_missions
                            WHERE user_id = :uid AND week_start = :w AND mission_key = :k
                            LIMIT 1');
  $rowStmt->execute(['uid' => $me['id'], 'w' => $weekStart, 'k' => $missionKey]);

  $m = $rowStmt->fetch();
  if (!$m) {
    $pdo->rollBack();
    json_response(['success' => false, 'error' => 'Mission not found'], 404);
  }

  if (!empty($m['claimed_at'])) {
    $pdo->rollBack();
    json_response(['success' => false, 'error' => 'Already claimed'], 409);
  }

  $progress = (int)($m['progress'] ?? 0);
  $target = (int)($m['target'] ?? 0);
  if ($target <= 0 || $progress < $target) {
    $pdo->rollBack();
    json_response(['success' => false, 'error' => 'Mission is not complete yet'], 422);
  }

  // Mark claimed
  $pdo->prepare('UPDATE weekly_missions SET claimed_at = UTC_TIMESTAMP()
                 WHERE user_id = :uid AND week_start = :w AND mission_key = :k')
    ->execute(['uid' => $me['id'], 'w' => $weekStart, 'k' => $missionKey]);

  $rewardXp = (int)($m['reward_xp'] ?? 0);
  $rewardFz = (int)($m['reward_freezes'] ?? 0);

  // Award XP (best-effort)
  $xpNew = null;
  $levelNew = null;
  $levelUp = false;
  try {
    $g = $pdo->prepare('SELECT xp, level, daily_goal, badges_json FROM user_gamification WHERE user_id = :uid LIMIT 1');
    $g->execute(['uid' => $me['id']]);
    $gr = $g->fetch();

    $xp = (int)($gr['xp'] ?? 0);
    $level = max(1, (int)($gr['level'] ?? 1));
    $dailyGoal = max(5, min(200, (int)($gr['daily_goal'] ?? 10)));
    $badgesJson = (string)($gr['badges_json'] ?? '[]');

    $xpNew = max(0, $xp + max(0, $rewardXp));
    $levelNew = (int)floor($xpNew / 250) + 1;
    $levelUp = $levelNew > $level;

    if ($gr) {
      $pdo->prepare('UPDATE user_gamification SET xp = :xp, level = :lvl, updated_at = UTC_TIMESTAMP() WHERE user_id = :uid')
        ->execute(['xp' => $xpNew, 'lvl' => $levelNew, 'uid' => $me['id']]);
    } else {
      $pdo->prepare('INSERT INTO user_gamification (user_id, daily_goal, xp, level, badges_json, updated_at)
                     VALUES (:uid, :g, :xp, :lvl, :b, UTC_TIMESTAMP())')
        ->execute([
          'uid' => $me['id'],
          'g' => $dailyGoal,
          'xp' => $xpNew,
          'lvl' => $levelNew,
          'b' => $badgesJson,
        ]);
    }
  } catch (Throwable $e) {
    // ignore
  }

  // Award streak freezes
  $streakFreezes = 0;
  try {
    $pdo->prepare('UPDATE user_progress SET streak_freezes = streak_freezes + :inc WHERE user_id = :uid')
      ->execute(['inc' => max(0, $rewardFz), 'uid' => $me['id']]);

    $p = $pdo->prepare('SELECT streak_freezes FROM user_progress WHERE user_id = :uid LIMIT 1');
    $p->execute(['uid' => $me['id']]);
    $streakFreezes = (int)(($p->fetch() ?: [])['streak_freezes'] ?? 0);
  } catch (Throwable $e) {
    $streakFreezes = 0;
  }

  $pdo->commit();

  // Return updated missions + snapshot
  $missions = missions_list($pdo, (string)$me['id'], $weekStart);
  $claimable = missions_claimable_count($pdo, (string)$me['id'], $weekStart);

  json_response(['success' => true, 'data' => [
    'weekStart' => $weekStart,
    'missions' => $missions,
    'claimable' => $claimable,
    'rewards' => [
      'xp' => max(0, $rewardXp),
      'streakFreezes' => max(0, $rewardFz),
    ],
    'progress' => array_filter([
      'xp' => $xpNew,
      'level' => $levelNew,
      'levelUp' => $levelUp,
      'streakFreezes' => $streakFreezes,
      'missionsClaimable' => $claimable,
    ], fn($v) => $v !== null),
  ]]);

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  json_response([
    'success' => false,
    'error' => 'Server error',
    'detail' => $e->getMessage(),
    'where' => basename((string)$e->getFile()) . ':' . (string)$e->getLine(),
  ], 500);
}
