<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/_helpers.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}
$me = require_auth($pdo);
$body = read_json();
require_fields($body, ['groupId', 'title', 'courseId']);

$groupId = trim((string)$body['groupId']);
$title = trim((string)$body['title']);
$courseId = trim((string)$body['courseId']);
$count = isset($body['count']) ? (int)$body['count'] : 20;
$duration = isset($body['durationMins']) ? (int)$body['durationMins'] : 20;

if ($groupId === '' || $title === '' || $courseId === '') json_response(['success' => false, 'error' => 'Invalid fields'], 422);
$count = max(5, min(60, $count));
$duration = max(5, min(120, $duration));

group_require_owner($pdo, $groupId, $me);

// Ensure course exists
$c = $pdo->prepare('SELECT id, code, title, level FROM courses WHERE id = :id LIMIT 1');
$c->execute(['id' => $courseId]);
$course = $c->fetch();
if (!$course) json_response(['success' => false, 'error' => 'Course not found'], 404);

// Pull random questions for the course
$q = $pdo->prepare('
  SELECT q.id, q.bank_id AS bankId, q.prompt, q.options
  FROM questions q
  JOIN banks b ON b.id = q.bank_id
  WHERE b.course_id = :cid
  ORDER BY RAND()
  LIMIT ' . (int)$count . '
');
$q->execute(['cid' => $courseId]);
$qs = $q->fetchAll() ?: [];
if (count($qs) === 0) json_response(['success' => false, 'error' => 'No questions found for this course'], 404);

$challengeId = uuidv4();

try {
  $pdo->beginTransaction();

  $pdo->prepare('
    INSERT INTO study_group_challenges (id, group_id, title, course_id, total_questions, duration_minutes, created_by, created_at)
    VALUES (:id, :gid, :t, :cid, :n, :d, :uid, UTC_TIMESTAMP())
  ')->execute([
    'id' => $challengeId,
    'gid' => $groupId,
    't' => $title,
    'cid' => $courseId,
    'n' => count($qs),
    'd' => $duration,
    'uid' => $me['id'],
  ]);

  $ins = $pdo->prepare('
    INSERT INTO study_group_challenge_questions (challenge_id, question_id, bank_id, position)
    VALUES (:cid, :qid, :bid, :pos)
  ');
  foreach ($qs as $i => $row) {
    $ins->execute([
      'cid' => $challengeId,
      'qid' => (string)$row['id'],
      'bid' => (string)$row['bankId'],
      'pos' => $i,
    ]);
  }

  $pdo->commit();
} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  json_response(['success' => false, 'error' => 'Failed to create challenge'], 500);
}

json_response([
  'success' => true,
  'data' => [
    'challenge' => [
      'id' => $challengeId,
      'groupId' => $groupId,
      'title' => $title,
      'courseId' => $courseId,
      'courseCode' => (string)($course['code'] ?? ''),
      'courseTitle' => (string)($course['title'] ?? ''),
      'durationMins' => $duration,
      'total' => count($qs),
      'createdAt' => now_iso(),
    ]
  ]
]);
