<?php
// public_html/jabuspark_api/helpers.php
declare(strict_types=1);

/**
 * CORS HEADERS
 * Must run BEFORE any output
 */
if (!function_exists('cors_headers')) {

function cors_headers(): void {
  $allowedOrigins = [
    'https://jabuspark-mvp1.vercel.app',
    'https://jabuspark-mvp1.vercel.app/',
    'https://jabustudyhub.com.ng',
    'https://www.jabustudyhub.com.ng',
    'http://localhost:5173',
    'http://127.0.0.1:5173',
  ];

  $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
  if (in_array($origin, $allowedOrigins, true)) {
    header("Access-Control-Allow-Origin: $origin");
    header("Access-Control-Allow-Credentials: true");
    header("Vary: Origin");
  }

  header('Access-Control-Allow-Methods: GET, POST, PUT, PATCH, DELETE, OPTIONS');
  header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With, Accept, Origin');
  header('Access-Control-Max-Age: 86400');
}
}


/**
 * Utility helpers
 */

function now_iso(): string {
  return gmdate('c');
}

function today_iso_date(): string {
  return gmdate('Y-m-d');
}

function uuidv4(): string {
  $data = random_bytes(16);
  $data[6] = chr((ord($data[6]) & 0x0f) | 0x40);
  $data[8] = chr((ord($data[8]) & 0x3f) | 0x80);
  $hex = bin2hex($data);
  return sprintf(
    '%s-%s-%s-%s-%s',
    substr($hex, 0, 8),
    substr($hex, 8, 4),
    substr($hex, 12, 4),
    substr($hex, 16, 4),
    substr($hex, 20, 12)
  );
}

function require_fields(array $body, array $fields): void {
  foreach ($fields as $f) {
    if (!array_key_exists($f, $body) || $body[$f] === null || $body[$f] === '') {
      json_response([
        'success' => false,
        'error' => 'Missing field: ' . $f
      ], 422);
    }
  }
}

function bearer_token(): ?string {
  $h = $_SERVER['HTTP_AUTHORIZATION'] ?? $_SERVER['Authorization'] ?? '';
  if (!$h) return null;
  if (stripos($h, 'Bearer ') === 0) {
    return trim(substr($h, 7));
  }
  return null;
}

function public_url(string $pathFromApiRoot): string {
  $base = rtrim((string)PUBLIC_BASE, '/');
  if ($base === '') return $pathFromApiRoot;
  return $base . $pathFromApiRoot;
}

function safe_filename(string $name): string {
  $name = preg_replace('/[^A-Za-z0-9._-]+/', '_', $name) ?? 'file';
  $name = trim($name, '_');
  return $name === '' ? 'file' : $name;
}

function json_get(?string $json, $fallback) {
  if (!$json) return $fallback;
  $d = json_decode($json, true);
  return is_array($d) || is_object($d) ? $d : $fallback;
}

/**
 * Upload audit log (optional; created automatically if missing)
 */
function ensure_upload_audit_table(PDO $pdo): void {
  $pdo->exec('CREATE TABLE IF NOT EXISTS upload_audit (
    id CHAR(36) NOT NULL,
    action VARCHAR(16) NOT NULL,
    entity_type VARCHAR(32) NOT NULL,
    entity_id CHAR(36) NOT NULL,
    course_id VARCHAR(24) NULL,
    file_path VARCHAR(255) NULL,
    actor_id CHAR(36) NULL,
    target_user_id CHAR(36) NULL,
    details_json JSON NULL,
    created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    INDEX idx_upload_audit_created (created_at),
    INDEX idx_upload_audit_course (course_id),
    INDEX idx_upload_audit_entity (entity_type, entity_id),
    INDEX idx_upload_audit_actor (actor_id),
    INDEX idx_upload_audit_target (target_user_id)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci');
}

function upload_audit_log(PDO $pdo, array $params): void {
  // params: action, entityType, entityId, courseId?, filePath?, actorId?, targetUserId?, details?
  try {
    ensure_upload_audit_table($pdo);
    $id = uuidv4();
    $details = $params['details'] ?? null;
    $detailsJson = null;
    if (is_array($details) || is_object($details)) {
      $detailsJson = json_encode($details, JSON_UNESCAPED_UNICODE);
    }
    $stmt = $pdo->prepare('INSERT INTO upload_audit (id, action, entity_type, entity_id, course_id, file_path, actor_id, target_user_id, details_json, created_at)
                           VALUES (:id,:a,:et,:eid,:cid,:fp,:actor,:target,:dj,UTC_TIMESTAMP())');
    $stmt->execute([
      ':id' => $id,
      ':a' => (string)($params['action'] ?? ''),
      ':et' => (string)($params['entityType'] ?? ''),
      ':eid' => (string)($params['entityId'] ?? ''),
      ':cid' => ($params['courseId'] ?? null) !== '' ? $params['courseId'] : null,
      ':fp' => ($params['filePath'] ?? null) !== '' ? $params['filePath'] : null,
      ':actor' => ($params['actorId'] ?? null) !== '' ? $params['actorId'] : null,
      ':target' => ($params['targetUserId'] ?? null) !== '' ? $params['targetUserId'] : null,
      ':dj' => $detailsJson,
    ]);
  } catch (Throwable $e) {
    // best-effort only
  }
}
