<?php
// public_html/api/index.php
declare(strict_types=1);

require_once __DIR__ . '/response.php';
require_once __DIR__ . '/helpers.php';

// Always set CORS headers
cors_headers();

// Preflight should NOT touch DB
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') {
  http_response_code(204);
  exit;
}

// DB only for real requests
require_once __DIR__ . '/config.php';

$method = strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET'));

// Request path (no querystring)
$uri  = (string)($_SERVER['REQUEST_URI'] ?? '/');
$path = (string)(parse_url($uri, PHP_URL_PATH) ?: '/');

// Base folder of this script (e.g. "/api")
$base = rtrim(dirname((string)($_SERVER['SCRIPT_NAME'] ?? '')), '/');
if ($base !== '' && $base !== '.' && strpos($path, $base) === 0) {
  $path = substr($path, strlen($base));
}
$path = '/' . ltrim($path, '/');

// ✅ Support both "/api/xxx" and "/xxx"
if (strpos($path, '/api/') === 0) {
  $path = substr($path, 4); // remove "/api"
  $path = '/' . ltrim($path, '/');
}

// Sub path becomes the route matcher
$sub = '/' . trim($path, '/');
if ($sub === '/') $sub = '/';

$routes = [
  // auth
  'POST /auth/register' => __DIR__ . '/auth/register.php',
  'POST /auth/login'    => __DIR__ . '/auth/login.php',
  'POST /auth/logout'   => __DIR__ . '/auth/logout.php',

  // health
  'GET /health'          => __DIR__ . '/health.php',

  // me
  'GET /me'             => __DIR__ . '/me/get.php',
  'PATCH /me/profile'   => __DIR__ . '/me/patch_profile.php',

  // onboarding
  'POST /onboarding/department-request' => __DIR__ . '/onboarding/department_request.php',

  // catalog
  'GET /catalog/faculties'   => __DIR__ . '/catalog/faculties.php',
  'GET /catalog/departments' => __DIR__ . '/catalog/departments.php',
  'GET /catalog/courses'     => __DIR__ . '/catalog/courses.php',
  // catalog (special)
  'GET /catalog/gns-courses' => __DIR__ . '/catalog/gns_courses.php',

  // materials
  'GET /materials'      => __DIR__ . '/materials/list.php',
  'POST /materials'     => __DIR__ . '/materials/create.php',
  'DELETE /materials'   => __DIR__ . '/materials/delete.php',
  // ✅ NEW (for perfect uploads workflow)
  'PATCH /materials'    => __DIR__ . '/materials/patch.php', // implement this file next

  // past questions
  'GET /pastquestions'     => __DIR__ . '/pastquestions/list.php',
  'POST /pastquestions'    => __DIR__ . '/pastquestions/create.php',
  'DELETE /pastquestions'  => __DIR__ . '/pastquestions/delete.php',
  // ✅ NEW (for perfect uploads workflow)
  'PATCH /pastquestions'   => __DIR__ . '/pastquestions/patch.php', // implement this file next

  // ✅ NEW: “My uploads” endpoints (used by Uploads.vue manage section)
  // These should enforce auth + return ONLY uploads created_by=me.id (or admin sees all).
  'GET /uploader/pastquestions' => __DIR__ . '/uploader/pastquestions.php', // implement this file next
  'GET /uploader/materials'     => __DIR__ . '/uploader/materials.php',     // implement this file next

  // banks
  'GET /banks'          => __DIR__ . '/banks/list.php',
  'GET /banks/get'      => __DIR__ . '/banks/get.php',
  'POST /banks'         => __DIR__ . '/banks/create.php',
  'DELETE /banks'       => __DIR__ . '/banks/delete.php',

  // practice
  'POST /practice/submit' => __DIR__ . '/practice/submit.php',
  'POST /practice/reset'  => __DIR__ . '/practice/reset.php',
  'GET /practice/review'  => __DIR__ . '/practice/review.php',

  // theory (writing practice)
  'POST /theory/submit'   => __DIR__ . '/theory/submit.php',
  'GET /theory/attempts'  => __DIR__ . '/theory/attempts.php',
  'POST /theory/reset'    => __DIR__ . '/theory/reset.php',

  // progress
  'GET /progress'         => __DIR__ . '/progress/get.php',
  'GET /progress/courses' => __DIR__ . '/progress/courses.php',
  'GET /progress/trend'   => __DIR__ . '/progress/trend.php',

  // gamification
  'POST /gamification/goal'            => __DIR__ . '/gamification/goal.php',
  'GET /gamification/missions'         => __DIR__ . '/gamification/missions.php',
  'POST /gamification/missions/claim'  => __DIR__ . '/gamification/missions_claim.php',

  // leaderboards
  'GET /leaderboard/weekly' => __DIR__ . '/leaderboard/weekly.php',

  // study groups + challenges
  'GET /groups/my'              => __DIR__ . '/groups/my.php',
  'GET /groups/badge'           => __DIR__ . '/groups/badge.php',
  'POST /groups/create'         => __DIR__ . '/groups/create.php',
  'POST /groups/join'           => __DIR__ . '/groups/join.php',
  'GET /groups/get'             => __DIR__ . '/groups/get.php',
  'POST /groups/leave'          => __DIR__ . '/groups/leave.php',
  'POST /groups/challenge/create' => __DIR__ . '/groups/challenge_create.php',
  'GET /groups/challenge/list'  => __DIR__ . '/groups/challenge_list.php',
  'GET /challenge/get'          => __DIR__ . '/challenge/get.php',
  'POST /challenge/submit'      => __DIR__ . '/challenge/submit.php',
  'GET /challenge/scoreboard'   => __DIR__ . '/challenge/scoreboard.php',

  // 1v1 Duel (shareable challenge links)
  'POST /duel/create' => __DIR__ . '/duel/create.php',
  'GET /duel/get'     => __DIR__ . '/duel/get.php',
  'POST /duel/join'   => __DIR__ . '/duel/join.php',
  'GET /duel/questions' => __DIR__ . '/duel/questions.php',
  'GET /duel/start'   => __DIR__ . '/duel/start.php',
  'POST /duel/submit' => __DIR__ . '/duel/submit.php',
  'GET /duel/result'  => __DIR__ . '/duel/result.php',

  // notifications / announcements (JabuNotify)
  'GET /notify/channels' => __DIR__ . '/notify/channels.php',
  'POST /notify/follow'  => __DIR__ . '/notify/follow.php',
  'GET /notify/feed'     => __DIR__ . '/notify/feed.php',
  'POST /notify/read'    => __DIR__ . '/notify/read.php',
  'POST /notify/post'    => __DIR__ . '/notify/post.php',

  // exam mode
  'GET /exam/start'   => __DIR__ . '/exam/start.php',
  'GET /exam/get'     => __DIR__ . '/exam/get.php',
  'GET /exam/result'  => __DIR__ . '/exam/result.php',
  'POST /exam/submit' => __DIR__ . '/exam/submit.php',

  // saved items
  'POST /save/toggle' => __DIR__ . '/save/toggle.php',

  // ✅ Saved item details (for Saved tab without downloading entire catalogs)
  'GET /saved/pastquestions' => __DIR__ . '/saved/pastquestions.php',
  'GET /saved/materials'     => __DIR__ . '/saved/materials.php',
  'GET /saved/questions'     => __DIR__ . '/saved/questions.php',

  // course rep requests
  'POST /rep/request' => __DIR__ . '/rep/request.php',
  'GET /rep/my'       => __DIR__ . '/rep/my.php',

  // AI (Gemini)
  'POST /ai/explain-mcq'   => __DIR__ . '/ai/explain_mcq.php',
  'POST /ai/generate-bank' => __DIR__ . '/ai/generate_bank.php',
  'POST /ai/doc-chat'      => __DIR__ . '/ai/doc_chat.php',

  // AI coach (theory writing)
  'POST /ai/theory-coach'      => __DIR__ . '/ai/theory_coach.php',

  // admin
  'GET /admin/rep-requests'          => __DIR__ . '/admin/rep_requests.php',
  'POST /admin/rep-requests/approve' => __DIR__ . '/admin/rep_requests_approve.php',
  'POST /admin/rep-requests/deny'    => __DIR__ . '/admin/rep_requests_deny.php',
  'POST /admin/ai/fill-explanations' => __DIR__ . '/admin/ai_fill_explanations.php',

  // Admin AI tools for theory banks
  'POST /admin/ai/fill-theory-guides' => __DIR__ . '/admin/ai_fill_theory_guides.php',
  'POST /admin/ai/quality-audit'      => __DIR__ . '/admin/ai_quality_audit.php',

  // ✅ NEW: Admin course reps management (used by AdminCourseReps.vue)
  'GET /admin/course-reps'                 => __DIR__ . '/admin/course_reps.php',                 // implement next
  'POST /admin/course-reps/assign'         => __DIR__ . '/admin/course_reps_assign.php',          // implement next
  'POST /admin/course-reps/unassign'       => __DIR__ . '/admin/course_reps_unassign.php',        // implement next
  'POST /admin/course-reps/uploads-disabled' => __DIR__ . '/admin/course_reps_uploads_disabled.php', // implement next
  'POST /admin/course-reps/suspend'        => __DIR__ . '/admin/course_reps_suspend.php',         // implement next

  // ✅ NEW: Upload audit log (used by AdminUploadLogs.vue)
  'GET /admin/upload-audit' => __DIR__ . '/admin/upload_audit.php', // implement next
];

// Friendly path: /banks/<id> -> /banks/get?id=<id>
$key = $method . ' ' . $sub;
if ($method === 'GET' && preg_match('#^/banks/([^/]+)$#', $sub, $mm)) {
  $_GET['id'] = urldecode($mm[1]);
  $key = 'GET /banks/get';
}

if (!isset($routes[$key])) {
  $debug = (getenv('JABUSPARK_DEBUG') === '1');
  if ($debug) {
    json_response([
      'success' => false,
      'error' => 'Not found',
      'debug' => [
        'method' => $method,
        'path' => $path,
        'sub' => $sub,
        'key' => $key
      ]
    ], 404);
  }
  json_response(['success' => false, 'error' => 'Not found'], 404);
}

require $routes[$key];
