<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'GET') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);

$courseId = isset($_GET['courseId']) ? trim((string)$_GET['courseId']) : '';
$limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
$limit = max(5, min(200, $limit));

// ISO week (Mon-Sun) in UTC
$weekKey = (new DateTimeImmutable('now', new DateTimeZone('UTC')))->format('o-\\WW');

$sql = "
  SELECT
    u.id AS userId,
    COALESCE(NULLIF(u.full_name, ''), NULLIF(u.email, ''), 'Student') AS name,
    COALESCE(SUM(CASE WHEN a.is_correct = 1 THEN 1 ELSE 0 END), 0) AS correct,
    COUNT(*) AS attempts,
    COALESCE(SUM(a.seconds_spent), 0) AS seconds
  FROM user_question_attempts a
  JOIN banks b ON b.id = a.bank_id
  JOIN users u ON u.id = a.user_id
  WHERE YEARWEEK(a.created_at, 1) = YEARWEEK(UTC_DATE(), 1)
    AND (:cid1 = '' OR b.course_id = :cid2)
  GROUP BY u.id
  ORDER BY correct DESC, attempts DESC, seconds ASC
  LIMIT " . (int)$limit;

$stmt = $pdo->prepare($sql);
$stmt->execute([
  'cid1' => $courseId,
  'cid2' => $courseId,
]);
$rows = $stmt->fetchAll() ?: [];

// Compute my stats + rank (best-effort, keep it light)
$my = [
  'userId' => (string)$me['id'],
  'name' => (string)($me['full_name'] ?? $me['email'] ?? 'Me'),
  'correct' => 0,
  'attempts' => 0,
  'seconds' => 0,
  'rank' => null,
];

// Find my entry in returned rows (if in top limit)
foreach ($rows as $idx => $r) {
  if ((string)$r['userId'] === (string)$me['id']) {
    $my['correct'] = (int)($r['correct'] ?? 0);
    $my['attempts'] = (int)($r['attempts'] ?? 0);
    $my['seconds'] = (int)($r['seconds'] ?? 0);
    $my['rank'] = $idx + 1;
    break;
  }
}

// If not in top list, compute my stats and rank by scanning a larger set.
if ($my['rank'] === null) {
  // My stats
  $s = $pdo->prepare('
    SELECT
      COALESCE(SUM(CASE WHEN a.is_correct = 1 THEN 1 ELSE 0 END), 0) AS correct,
      COUNT(*) AS attempts,
      COALESCE(SUM(a.seconds_spent), 0) AS seconds
    FROM user_question_attempts a
    JOIN banks b ON b.id = a.bank_id
    WHERE YEARWEEK(a.created_at, 1) = YEARWEEK(UTC_DATE(), 1)
      AND a.user_id = :uid
      AND (:cid1 = "" OR b.course_id = :cid2)
  ');
  $s->execute([
    'uid' => $me['id'],
    'cid1' => $courseId,
    'cid2' => $courseId,
  ]);
  $sr = $s->fetch() ?: [];
  $my['correct'] = (int)($sr['correct'] ?? 0);
  $my['attempts'] = (int)($sr['attempts'] ?? 0);
  $my['seconds'] = (int)($sr['seconds'] ?? 0);

  // Rank: count users strictly better by (correct desc, attempts desc, seconds asc)
  $rankQ = $pdo->prepare('
    SELECT COUNT(*) AS better
    FROM (
      SELECT
        a.user_id AS uid,
        COALESCE(SUM(CASE WHEN a.is_correct = 1 THEN 1 ELSE 0 END), 0) AS correct,
        COUNT(*) AS attempts,
        COALESCE(SUM(a.seconds_spent), 0) AS seconds
      FROM user_question_attempts a
      JOIN banks b ON b.id = a.bank_id
      WHERE YEARWEEK(a.created_at, 1) = YEARWEEK(UTC_DATE(), 1)
        AND (:cid1 = "" OR b.course_id = :cid2)
      GROUP BY a.user_id
    ) x
	    WHERE (x.correct > :c1)
	       OR (x.correct = :c2 AND x.attempts > :a1)
	       OR (x.correct = :c3 AND x.attempts = :a2 AND x.seconds < :sec)
  ');
  $rankQ->execute([
    'cid1' => $courseId,
    'cid2' => $courseId,
	    'c1' => $my['correct'],
	    'c2' => $my['correct'],
	    'c3' => $my['correct'],
	    'a1' => $my['attempts'],
	    'a2' => $my['attempts'],
    'sec' => $my['seconds'],
  ]);
  $better = (int)(($rankQ->fetch()['better'] ?? 0));
  $my['rank'] = $better + 1;
}

// Normalize rows
$out = [];
foreach ($rows as $i => $r) {
  $out[] = [
    'rank' => $i + 1,
    'userId' => (string)$r['userId'],
    'name' => (string)$r['name'],
    'correct' => (int)($r['correct'] ?? 0),
    'attempts' => (int)($r['attempts'] ?? 0),
    'seconds' => (int)($r['seconds'] ?? 0),
  ];
}

json_response([
  'success' => true,
  'data' => [
    'week' => $weekKey,
    'courseId' => $courseId,
    'items' => $out,
    'me' => $my,
  ]
]);
