<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';
$notify_helpers = __DIR__ . '/../notify/_helpers.php';
if (is_file($notify_helpers)) { require_once $notify_helpers; }
require_once __DIR__ . '/../ai/_extract.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') json_response(['success'=>false,'error'=>'Method not allowed'],405);

$me = require_auth($pdo);
// allow admin or approved course rep assigned to this course

$title = trim((string)($_POST['title'] ?? ''));
$courseId = trim((string)($_POST['courseId'] ?? ''));
$type = trim((string)($_POST['type'] ?? 'pdf'));
$tagsRaw = $_POST['tags'] ?? '[]';

if ($title === '' || $courseId === '') json_response(['success'=>false,'error'=>'title and courseId are required'],422);

require_uploader($pdo, $me, $courseId);

// Resolve course code for nicer announcements
$courseCode = $courseId;
try {
  $cs = $pdo->prepare('SELECT code FROM courses WHERE id = :cid LIMIT 1');
  $cs->execute([':cid' => $courseId]);
  $cc = $cs->fetchColumn();
  if ($cc) $courseCode = strtoupper((string)$cc);
} catch (Throwable $e) { /* ignore */ }

$tags = [];
if (is_string($tagsRaw)) {
  $decoded = json_decode($tagsRaw, true);
  if (is_array($decoded)) $tags = $decoded;
  else $tags = array_values(array_filter(array_map('trim', explode(',', $tagsRaw))));
} elseif (is_array($tagsRaw)) {
  $tags = $tagsRaw;
}

if (!isset($_FILES['file']) || ($_FILES['file']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) {
  json_response(['success'=>false,'error'=>'file is required (multipart form-data field: file)'],422);
}

$f = $_FILES['file'];
$orig = (string)($f['name'] ?? 'file.pdf');
$tmp = (string)($f['tmp_name'] ?? '');
$size = (int)($f['size'] ?? 0);

if ($size <= 0) json_response(['success'=>false,'error'=>'Empty file'],422);
if ($size > 50 * 1024 * 1024) json_response(['success'=>false,'error'=>'File too large (max 50MB)'],413);

$ext = strtolower(pathinfo($orig, PATHINFO_EXTENSION));
if ($ext === '') $ext = 'pdf';

// Security: allow only safe file types
$allowed = ['pdf','png','jpg','jpeg','webp','doc','docx'];
if (!in_array($ext, $allowed, true)) {
  json_response(['success'=>false,'error'=>'Unsupported file type'],422);
}

// Normalize type based on extension (server is source of truth)
if (in_array($ext, ['png','jpg','jpeg','webp'], true)) $type = 'image';
else if (in_array($ext, ['doc','docx'], true)) $type = 'doc';
else $type = 'pdf';

// Best-effort MIME sniff (prevents "rename a .php to .pdf")
$ok = true;
try {
  $fi = new finfo(FILEINFO_MIME_TYPE);
  $mime = $fi->file($tmp);
  if (is_string($mime)) {
    if ($type === 'pdf' && $mime !== 'application/pdf') $ok = false;
    if ($type === 'image' && !preg_match('#^image/#', $mime)) $ok = false;
    // doc/docx MIME varies on some hosts; don't hard-fail, just allow.
  }
} catch (Throwable $e) { /* ignore */ }
if (!$ok) json_response(['success'=>false,'error'=>'File content does not match type'],422);

$dir = UPLOAD_DIR . '/materials';
if (!is_dir($dir)) @mkdir($dir, 0775, true);

$fname = safe_filename(pathinfo($orig, PATHINFO_FILENAME));
$final = $fname . '_' . substr(hash('sha1', random_bytes(16)), 0, 10) . '.' . $ext;

$dest = $dir . '/' . $final;
if (!move_uploaded_file($tmp, $dest)) {
  json_response(['success'=>false,'error'=>'Failed to save file'],500);
}

$id = uuidv4();
$relPath = '/uploads/materials/' . $final;

$stmt = $pdo->prepare('INSERT INTO materials (id, course_id, title, type, file_path, uploaded_at, tags, created_by)
                       VALUES (:id,:c,:t,:ty,:fp,UTC_DATE(),:tags,:cb)');
$stmt->execute([
  ':id'=>$id, ':c'=>$courseId, ':t'=>$title, ':ty'=>$type, ':fp'=>$relPath,
  ':tags'=>json_encode(array_values($tags), JSON_UNESCAPED_UNICODE),
  ':cb'=>$me['id']
]);

// audit (best-effort)
upload_audit_log($pdo, [
  'action' => 'create',
  'entityType' => 'material',
  'entityId' => $id,
  'courseId' => $courseId,
  'filePath' => $relPath,
  'actorId' => (string)$me['id'],
  'targetUserId' => (string)$me['id'],
  'details' => ['title' => $title, 'type' => $type]
]);

// Best-effort PDF text extraction for Ask-AI (optional)
try {
  $canExtract = ($ext === 'pdf' || strtolower($type) === 'pdf');
  if ($canExtract) {
    $extracted = pdf_to_text_best_effort($dest);
    if (is_string($extracted) && trim($extracted) !== '') {
      // Save only if the columns exist (migration may not be run yet)
      $u = $pdo->prepare('UPDATE materials SET extracted_text = :t, extracted_at = UTC_TIMESTAMP() WHERE id = :id');
      $u->execute([':t' => $extracted, ':id' => $id]);
    }
  }
} catch (Throwable $e) {
  // ignore
}

// Best-effort: create an announcement post for this upload
try {
  if (function_exists('notify_ensure_course_channel')) {
    $ch = notify_ensure_course_channel($pdo, $courseId, $me['id']);
    if ($ch && isset($ch['id'])) {
      $pid = uuidv4();
      $ptitle = 'New material: ' . $title;
      $pbody = 'A new material was uploaded for ' . $courseCode . '. Tap to open.';
      $plink = public_url($relPath);
      $ins = $pdo->prepare('INSERT INTO notify_posts (id, channel_id, title, body, link_url, attachment_url, attachment_type, is_pinned, created_by) VALUES (:id,:cid,:t,:b,:l,:a,:at,0,:uid)');
      $ins->execute([
        ':id' => $pid,
        ':cid' => (string)$ch['id'],
        ':t' => $ptitle,
        ':b' => $pbody,
        ':l' => $plink,
        ':a' => null,
        ':at' => null,
        ':uid' => $me['id'],
      ]);
    }
  }
} catch (Throwable $e) {
  // ignore
}

json_response(['success'=>true,'data'=>[
  'material'=>[
    'id'=>$id,'courseId'=>$courseId,'title'=>$title,'type'=>$type,'fileUrl'=>public_url($relPath),
    'uploadedAt'=>gmdate('Y-m-d'),'tags'=>$tags
  ]
]],201);
