<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'PATCH') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$body = read_json();
require_fields($body, ['id']);

$id = trim((string)($body['id'] ?? ''));
if ($id === '') {
  json_response(['success' => false, 'error' => 'Missing id'], 422);
}

// Fetch existing
$stmt = $pdo->prepare('SELECT id, course_id, title, type, tags, file_path, created_by FROM materials WHERE id = :id LIMIT 1');
$stmt->execute([':id' => $id]);
$row = $stmt->fetch();
if (!$row) {
  json_response(['success' => false, 'error' => 'Not found'], 404);
}

$role = (string)($me['role'] ?? 'student');
if ($role !== 'admin') {
  if ($role !== 'course_rep' || (string)$row['created_by'] !== (string)$me['id']) {
    json_response(['success' => false, 'error' => 'Forbidden'], 403);
  }
  // also validates uploads-disabled + course assignment
  require_uploader($pdo, $me, (string)$row['course_id']);
}

$updates = [];
$params = [':id' => $id];

if (array_key_exists('title', $body)) {
  $title = trim((string)($body['title'] ?? ''));
  if ($title === '') json_response(['success' => false, 'error' => 'title cannot be empty'], 422);
  $updates[] = 'title = :title';
  $params[':title'] = $title;
}

if (array_key_exists('type', $body)) {
  $type = trim((string)($body['type'] ?? ''));
  if ($type === '') json_response(['success' => false, 'error' => 'type cannot be empty'], 422);
  $updates[] = 'type = :type';
  $params[':type'] = $type;
}

if (array_key_exists('tags', $body)) {
  $tags = $body['tags'] ?? [];
  if (!is_array($tags)) $tags = [];
  $tags = array_values(array_unique(array_filter(array_map(fn($t) => trim((string)$t), $tags))));
  $updates[] = 'tags = :tags';
  $params[':tags'] = json_encode($tags, JSON_UNESCAPED_UNICODE);
}

if (count($updates) === 0) {
  json_response(['success' => false, 'error' => 'No fields to update'], 422);
}

$pdo->prepare('UPDATE materials SET ' . implode(', ', $updates) . ' WHERE id = :id')->execute($params);

$stmt = $pdo->prepare('SELECT id, course_id AS courseId, title, type, file_path, uploaded_at AS uploadedAt, tags, created_by AS createdBy
                        FROM materials WHERE id = :id LIMIT 1');
$stmt->execute([':id' => $id]);
$u = $stmt->fetch();

$item = $u ?: [];
$item['fileUrl'] = public_url((string)($item['file_path'] ?? ''));
unset($item['file_path']);
$item['tags'] = json_get($item['tags'] ?? null, []);

upload_audit_log($pdo, [
  'action' => 'update',
  'entityType' => 'material',
  'entityId' => $id,
  'courseId' => (string)$row['course_id'],
  'filePath' => (string)$row['file_path'],
  'actorId' => (string)$me['id'],
  'targetUserId' => (string)$row['created_by'],
  'details' => [
    'changed' => array_keys($params),
  ],
]);

json_response(['success' => true, 'data' => ['material' => $item]]);
