<?php
// public_html/api/notify/_helpers.php
declare(strict_types=1);

require_once __DIR__ . '/../helpers.php';

function notify_slug(string $s): string {
  $s = strtolower(trim($s));
  $s = preg_replace('/[^a-z0-9]+/', '-', $s) ?? '';
  $s = trim($s, '-');
  return $s !== '' ? $s : 'channel';
}

/**
 * Ensure base channels exist (general + academics).
 */
function notify_ensure_defaults(PDO $pdo): void {
  $defaults = [
    ['id' => '00000000-0000-4000-8000-000000000001', 'name' => 'General', 'slug' => 'general'],
    ['id' => '00000000-0000-4000-8000-000000000002', 'name' => 'Academics', 'slug' => 'academics'],
  ];

  foreach ($defaults as $c) {
    $stmt = $pdo->prepare('SELECT id FROM notify_channels WHERE id = :id LIMIT 1');
    $stmt->execute([':id' => $c['id']]);
    $exists = (bool)$stmt->fetchColumn();
    if ($exists) continue;

    $ins = $pdo->prepare('INSERT INTO notify_channels (id, name, slug, scope_type, scope_id, is_active)
                          VALUES (:id, :name, :slug, "general", NULL, 1)');
    $ins->execute([':id' => $c['id'], ':name' => $c['name'], ':slug' => $c['slug']]);
  }
}

/**
 * Ensure a course channel exists for a given course_id.
 * Returns channel row (id, name, slug).
 */
function notify_ensure_course_channel(PDO $pdo, string $courseId, ?string $createdBy = null): ?array {
  $courseId = trim($courseId);
  if ($courseId === '') return null;

  // Check course exists
  $c = $pdo->prepare('SELECT id, code FROM courses WHERE id = :cid LIMIT 1');
  $c->execute([':cid' => $courseId]);
  $course = $c->fetch();
  if (!$course) return null;

  $stmt = $pdo->prepare('SELECT id, name, slug FROM notify_channels WHERE scope_type = "course" AND scope_id = :cid LIMIT 1');
  $stmt->execute([':cid' => $courseId]);
  $row = $stmt->fetch();
  if ($row) return $row;

  $id = uuidv4();
  $code = strtoupper((string)($course['code'] ?? $courseId));
  $name = $code . ' Updates';
  $slug = notify_slug('course-' . $courseId);

  $ins = $pdo->prepare('INSERT INTO notify_channels (id, name, slug, scope_type, scope_id, is_active, created_by)
                        VALUES (:id, :name, :slug, "course", :cid, 1, :cb)');
  $ins->execute([
    ':id' => $id,
    ':name' => $name,
    ':slug' => $slug,
    ':cid' => $courseId,
    ':cb' => $createdBy,
  ]);

  return ['id' => $id, 'name' => $name, 'slug' => $slug];
}

function notify_unread_count(PDO $pdo, string $userId): int {
  // unread posts from followed channels, excluding expired
  $sql = '
    SELECT COUNT(*) AS c
    FROM notify_channel_follows f
    JOIN notify_posts p ON p.channel_id = f.channel_id
    LEFT JOIN notify_post_reads r ON r.post_id = p.id AND r.user_id = f.user_id
    WHERE f.user_id = :uid
      AND r.post_id IS NULL
      AND (p.expires_at IS NULL OR p.expires_at > UTC_TIMESTAMP())
  ';
  $q = $pdo->prepare($sql);
  $q->execute([':uid' => $userId]);
  return (int)($q->fetchColumn() ?: 0);
}
