<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/_helpers.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'GET') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);

// Used later for auto-following a user's course channels.
$courseIds = [];

// Ensure base channels exist
try {
  notify_ensure_defaults($pdo);
} catch (Throwable $e) {
  // ignore
}

// Ensure course channels for user's courses
try {
  $stmt = $pdo->prepare('SELECT course_id FROM user_courses WHERE user_id = :uid');
  $stmt->execute([':uid' => $me['id']]);
  $courseIds = array_values(array_map(fn($r) => (string)$r['course_id'], $stmt->fetchAll() ?: []));
  foreach ($courseIds as $cid) {
    notify_ensure_course_channel($pdo, $cid, $me['id']);
  }
} catch (Throwable $e) {
  // ignore
}

// Auto-follow channels to make the feed work out of the box.
//
// Why:
// - The feed only shows posts from channels a user follows.
// - Students expect to see at least General/Academics immediately.
// - Students also expect to see updates for courses they are enrolled in.
//
// This is intentionally idempotent (INSERT IGNORE).

// 1) Always follow base channels
try {
  $pdo->prepare('INSERT IGNORE INTO notify_channel_follows (user_id, channel_id) VALUES (:uid, :gen), (:uid, :acad)')
      ->execute([
        ':uid' => $me['id'],
        ':gen' => '00000000-0000-4000-8000-000000000001',
        ':acad' => '00000000-0000-4000-8000-000000000002',
      ]);
} catch (Throwable $e) {
  // ignore
}

// 2) Auto-follow course channels for user's courses
try {
  if (!empty($courseIds)) {
    $placeholders = implode(',', array_fill(0, count($courseIds), '?'));
    $sqlFollow = 'INSERT IGNORE INTO notify_channel_follows (user_id, channel_id)
                 SELECT ?, c.id
                 FROM notify_channels c
                 WHERE c.is_active = 1
                   AND c.scope_type = "course"
                   AND c.scope_id IN (' . $placeholders . ')';
    $stmt = $pdo->prepare($sqlFollow);
    $stmt->execute(array_merge([(string)$me['id']], $courseIds));
  }
} catch (Throwable $e) {
  // ignore
}

// List channels + follow state + unread per channel
$sql = '
  SELECT
    c.id,
    c.name,
    c.slug,
    c.scope_type AS scopeType,
    c.scope_id AS scopeId,
    c.is_active AS isActive,
    CASE WHEN f.user_id IS NULL THEN 0 ELSE 1 END AS isFollowed,
    (
      SELECT COUNT(*)
      FROM notify_posts p
      LEFT JOIN notify_post_reads r
        ON r.post_id = p.id AND r.user_id = :uid_read
      WHERE p.channel_id = c.id
        AND r.post_id IS NULL
        AND (p.expires_at IS NULL OR p.expires_at > UTC_TIMESTAMP())
    ) AS unreadCount
  FROM notify_channels c
  LEFT JOIN notify_channel_follows f
    ON f.channel_id = c.id AND f.user_id = :uid_follow
  WHERE c.is_active = 1
  ORDER BY
    CASE
      WHEN c.slug = "general" THEN 0
      WHEN c.slug = "academics" THEN 1
      WHEN c.scope_type = "course" THEN 2
      ELSE 3
    END,
    c.name ASC
';

$stmt = $pdo->prepare($sql);
$stmt->execute([
  ':uid_read' => $me['id'],
  ':uid_follow' => $me['id'],
]);
$channels = $stmt->fetchAll() ?: [];

$unreadTotal = 0;
foreach ($channels as $c) {
  $unreadTotal += (int)($c['unreadCount'] ?? 0);
}

json_response([
  'success' => true,
  'data' => [
    'channels' => array_map(function ($c) {
      return [
        'id' => (string)$c['id'],
        'name' => (string)$c['name'],
        'slug' => (string)$c['slug'],
        'scopeType' => (string)$c['scopeType'],
        'scopeId' => $c['scopeId'] !== null ? (string)$c['scopeId'] : null,
        'isFollowed' => ((int)$c['isFollowed']) === 1,
        'unreadCount' => (int)($c['unreadCount'] ?? 0),
      ];
    }, $channels),
    'unreadTotal' => (int)$unreadTotal,
  ]
]);
