<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../auth.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'GET') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);

$limit = max(1, min(100, (int)($_GET['limit'] ?? 25)));
$channelId = trim((string)($_GET['channelId'] ?? ''));

$params = [':uid' => $me['id'], ':limit' => $limit];

$where = '';
if ($channelId !== '') {
  $where = ' AND c.id = :channelId ';
  $params[':channelId'] = $channelId;
}

// Feed = posts from channels the user follows.
$sql = '
  SELECT
    p.id,
    p.channel_id AS channelId,
    c.name AS channelName,
    c.slug AS channelSlug,
    p.title,
    p.body,
    p.link_url AS linkUrl,
    p.attachment_url AS attachmentUrl,
    p.attachment_type AS attachmentType,
    p.is_pinned AS isPinned,
    p.expires_at AS expiresAt,
    p.created_by AS createdBy,
    p.created_at AS createdAt,
    CASE WHEN r.post_id IS NULL THEN 0 ELSE 1 END AS isRead
  FROM notify_channel_follows f
  JOIN notify_channels c ON c.id = f.channel_id AND c.is_active = 1
  JOIN notify_posts p ON p.channel_id = c.id
  LEFT JOIN notify_post_reads r ON r.post_id = p.id AND r.user_id = f.user_id
  WHERE f.user_id = :uid
    AND (p.expires_at IS NULL OR p.expires_at > UTC_TIMESTAMP())
    ' . $where . '
  ORDER BY p.is_pinned DESC, p.created_at DESC
  LIMIT :limit
';

$stmt = $pdo->prepare($sql);
// MySQL needs explicit type for LIMIT when using named param
$stmt->bindValue(':uid', $me['id'], PDO::PARAM_STR);
$stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
if ($channelId !== '') {
  $stmt->bindValue(':channelId', $channelId, PDO::PARAM_STR);
}
$stmt->execute();
$rows = $stmt->fetchAll() ?: [];

$posts = array_map(function ($r) {
  return [
    'id' => (string)$r['id'],
    'channelId' => (string)$r['channelId'],
    'channelName' => (string)$r['channelName'],
    'channelSlug' => (string)$r['channelSlug'],
    'title' => (string)$r['title'],
    'body' => (string)$r['body'],
    'linkUrl' => $r['linkUrl'] !== null ? (string)$r['linkUrl'] : null,
    'attachmentUrl' => $r['attachmentUrl'] !== null ? (string)$r['attachmentUrl'] : null,
    'attachmentType' => $r['attachmentType'] !== null ? (string)$r['attachmentType'] : null,
    'isPinned' => ((int)$r['isPinned']) === 1,
    'expiresAt' => $r['expiresAt'] !== null ? (string)$r['expiresAt'] : null,
    'createdBy' => (string)$r['createdBy'],
    'createdAt' => (string)$r['createdAt'],
    'isRead' => ((int)$r['isRead']) === 1,
  ];
}, $rows);

json_response([
  'success' => true,
  'data' => [
    'posts' => $posts,
  ]
]);
