<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';
$notify_helpers = __DIR__ . '/../notify/_helpers.php';
if (is_file($notify_helpers)) { require_once $notify_helpers; }
require_once __DIR__ . '/../ai/_extract.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success'=>false,'error'=>'Method not allowed'],405);
}

$me = require_auth($pdo);

// Allow admin or approved course rep assigned to this course
$title = trim((string)($_POST['title'] ?? ''));
$titlePrefix = trim((string)($_POST['titlePrefix'] ?? ''));
$courseId = trim((string)($_POST['courseId'] ?? ''));
$session = trim((string)($_POST['session'] ?? ''));
$semester = trim((string)($_POST['semester'] ?? ''));

if ($courseId === '') {
  json_response(['success'=>false,'error'=>'courseId is required'],422);
}

require_uploader($pdo, $me, $courseId);

// Resolve course code for nicer announcements
$courseCode = $courseId;
try {
  $cs = $pdo->prepare('SELECT code FROM courses WHERE id = :cid LIMIT 1');
  $cs->execute([':cid' => $courseId]);
  $cc = $cs->fetchColumn();
  if ($cc) $courseCode = strtoupper((string)$cc);
} catch (Throwable $e) { /* ignore */ }

if ($session === '') $session = 'Unknown';
if ($semester === '') $semester = 'Unknown';

/**
 * Normalize PHP's $_FILES structure into an array of items:
 *   [['name'=>..., 'tmp_name'=>..., 'size'=>..., 'error'=>...], ...]
 */
function normalize_uploaded_files(array $files): array {
  $out = [];
  // If input looks like a single file
  if (isset($files['name']) && !is_array($files['name'])) {
    $out[] = $files;
    return $out;
  }
  // Multi-file
  $names = $files['name'] ?? [];
  $count = is_array($names) ? count($names) : 0;
  for ($i = 0; $i < $count; $i++) {
    $out[] = [
      'name' => $files['name'][$i] ?? '',
      'type' => $files['type'][$i] ?? '',
      'tmp_name' => $files['tmp_name'][$i] ?? '',
      'error' => $files['error'][$i] ?? UPLOAD_ERR_NO_FILE,
      'size' => $files['size'][$i] ?? 0,
    ];
  }
  return $out;
}

// Accept both "file" (single) and "files" / "file[]" (bulk)
$rawFiles = null;
if (isset($_FILES['files'])) {
  $rawFiles = $_FILES['files'];
} elseif (isset($_FILES['file'])) {
  $rawFiles = $_FILES['file'];
}

if (!$rawFiles) {
  json_response(['success'=>false,'error'=>'file is required (multipart form-data field: file or files[])'],422);
}

$files = normalize_uploaded_files((array)$rawFiles);

// Bulk is enabled if >1 file is provided
$isBulk = count($files) > 1;

// Titles can be provided as JSON array matching file order (optional)
$titles = [];
if (isset($_POST['titles'])) {
  $maybe = json_decode((string)$_POST['titles'], true);
  if (is_array($maybe)) $titles = $maybe;
}

// Validation: single upload still requires title (for backward compatibility)
if (!$isBulk && $title === '') {
  json_response(['success'=>false,'error'=>'title is required'],422);
}

$MAX_BYTES = 50 * 1024 * 1024;
$allowedExt = ['pdf','png','jpg','jpeg','webp'];

$dir = UPLOAD_DIR . '/pastquestions';
if (!is_dir($dir)) @mkdir($dir, 0775, true);

$results = [];
$errors = [];

// Hard cap to avoid accidental massive uploads
if (count($files) > 25) {
  json_response(['success'=>false,'error'=>'Too many files (max 25 per request)'],413);
}

foreach ($files as $idx => $f) {
  $errCode = (int)($f['error'] ?? UPLOAD_ERR_NO_FILE);
  if ($errCode !== UPLOAD_ERR_OK) {
    $errors[] = ['index'=>$idx,'name'=>(string)($f['name'] ?? ''),'error'=>'Upload error'];
    continue;
  }

  $orig = (string)($f['name'] ?? 'file.pdf');
  $tmp = (string)($f['tmp_name'] ?? '');
  $size = (int)($f['size'] ?? 0);

  if ($size <= 0) {
    $errors[] = ['index'=>$idx,'name'=>$orig,'error'=>'Empty file'];
    continue;
  }
  if ($size > $MAX_BYTES) {
    $errors[] = ['index'=>$idx,'name'=>$orig,'error'=>'File too large (max 50MB)'];
    continue;
  }

  $ext = strtolower(pathinfo($orig, PATHINFO_EXTENSION));
  if ($ext === '') $ext = 'pdf';
  if (!in_array($ext, $allowedExt, true)) {
    $errors[] = ['index'=>$idx,'name'=>$orig,'error'=>'Unsupported file type'];
    continue;
  }

  // Best-effort MIME sniff
  $ok = true;
  try {
    $fi = new finfo(FILEINFO_MIME_TYPE);
    $mime = $fi->file($tmp);
    if (is_string($mime)) {
      if ($ext === 'pdf' && $mime !== 'application/pdf') $ok = false;
      if (in_array($ext, ['png','jpg','jpeg','webp'], true) && !preg_match('#^image/#', $mime)) $ok = false;
    }
  } catch (Throwable $e) { /* ignore */ }

  if (!$ok) {
    $errors[] = ['index'=>$idx,'name'=>$orig,'error'=>'File content does not match type'];
    continue;
  }

  // Determine title
  $t = '';
  if (isset($titles[$idx]) && is_string($titles[$idx])) {
    $t = trim($titles[$idx]);
  }
  if ($t === '' && !$isBulk) {
    $t = $title;
  }
  if ($t === '') {
    $t = trim(pathinfo($orig, PATHINFO_FILENAME));
  }
  if ($titlePrefix !== '' && $isBulk) {
    $t = trim($titlePrefix . ' ' . $t);
  }
  if ($t === '') {
    $errors[] = ['index'=>$idx,'name'=>$orig,'error'=>'Missing title'];
    continue;
  }

  $fname = safe_filename(pathinfo($orig, PATHINFO_FILENAME));
  $final = $fname . '_' . substr(hash('sha1', random_bytes(16)), 0, 10) . '.' . $ext;
  $dest = $dir . '/' . $final;

  if (!move_uploaded_file($tmp, $dest)) {
    $errors[] = ['index'=>$idx,'name'=>$orig,'error'=>'Failed to save file'];
    continue;
  }

  $id = uuidv4();
  $relPath = '/uploads/pastquestions/' . $final;

  try {
    $stmt = $pdo->prepare('INSERT INTO past_questions (id, course_id, session, semester, title, file_path, uploaded_at, created_by)
                           VALUES (:id,:c,:s,:sem,:t,:fp,UTC_DATE(),:cb)');
    $stmt->execute([
      ':id'=>$id,
      ':c'=>$courseId,
      ':s'=>$session,
      ':sem'=>$semester,
      ':t'=>$t,
      ':fp'=>$relPath,
      ':cb'=>$me['id']
    ]);
  } catch (Throwable $e) {
    // If DB insert fails, keep file but report error
    $errors[] = ['index'=>$idx,'name'=>$orig,'error'=>'Database error'];
    continue;
  }

  // audit (best-effort)
  upload_audit_log($pdo, [
    'action' => 'create',
    'entityType' => 'past_question',
    'entityId' => $id,
    'courseId' => $courseId,
    'filePath' => $relPath,
    'actorId' => (string)$me['id'],
    'targetUserId' => (string)$me['id'],
    'details' => ['title' => $t, 'session' => $session, 'semester' => $semester]
  ]);

  // Best-effort PDF text extraction for Ask-AI (optional)
  try {
    if ($ext === 'pdf') {
      $extracted = pdf_to_text_best_effort($dest);
      if (is_string($extracted) && trim($extracted) !== '') {
        $u = $pdo->prepare('UPDATE past_questions SET extracted_text = :t, extracted_at = UTC_TIMESTAMP() WHERE id = :id');
        $u->execute([':t' => $extracted, ':id' => $id]);
      }
    }
  } catch (Throwable $e) { /* ignore */ }

  // Best-effort: create an announcement post for this upload
  try {
    if (function_exists('notify_ensure_course_channel')) {
      $ch = notify_ensure_course_channel($pdo, $courseId, $me['id']);
      if ($ch && isset($ch['id'])) {
        $pid = uuidv4();
        $ptitle = 'New past question: ' . $t;
        $pbody = 'A new past question was uploaded for ' . $courseCode . " (" . $session . " • " . $semester . "). Tap to open.";
        $plink = public_url($relPath);
        $ins = $pdo->prepare('INSERT INTO notify_posts (id, channel_id, title, body, link_url, attachment_url, attachment_type, is_pinned, created_by)
                              VALUES (:id,:cid,:t,:b,:l,:a,:at,0,:uid)');
        $ins->execute([
          ':id' => $pid,
          ':cid' => (string)$ch['id'],
          ':t' => $ptitle,
          ':b' => $pbody,
          ':l' => $plink,
          ':a' => null,
          ':at' => null,
          ':uid' => $me['id'],
        ]);
      }
    }
  } catch (Throwable $e) { /* ignore */ }

  $results[] = [
    'id' => $id,
    'courseId' => $courseId,
    'session' => $session,
    'semester' => $semester,
    'title' => $t,
    'fileUrl' => public_url($relPath),
    'uploadedAt' => gmdate('Y-m-d'),
  ];
}

if (!$isBulk) {
  if (!$results) {
    $msg = $errors[0]['error'] ?? 'Upload failed.';
    json_response(['success'=>false,'error'=>$msg],422);
  }
  json_response(['success'=>true,'data'=>['pastQuestion'=>$results[0]]],201);
}

// Bulk response (partial success possible)
json_response([
  'success' => count($results) > 0,
  'data' => [
    'uploaded' => $results,
    'failed' => $errors,
    'counts' => [
      'uploaded' => count($results),
      'failed' => count($errors),
      'total' => count($files),
    ]
  ]
], count($results) > 0 ? 200 : 422);
