<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$body = read_json();
require_fields($body, ['bankId', 'questionId', 'selectedIndex']);

$bankId = (string)$body['bankId'];
$qid = (string)$body['questionId'];
$selectedIndex = (int)$body['selectedIndex'];
$secondsSpent = isset($body['secondsSpent']) ? (int)$body['secondsSpent'] : 0;

/**
 * FIX: "explain" is treated as reserved keyword (EXPLAIN) in MariaDB/MySQL.
 * Use a safe alias instead (recommended), e.g. explanationText.
 */
$q = $pdo->prepare('
  SELECT
    q.id,
    q.prompt,
    q.options,
    q.answer_index AS answerIndex,
    q.explanation AS explanationText,
    b.course_id AS courseId
  FROM questions q
  JOIN banks b ON b.id = q.bank_id
  WHERE q.bank_id = :bid AND q.id = :id
  LIMIT 1
');
$q->execute(['bid' => $bankId, 'id' => $qid]);
$row = $q->fetch();
if (!$row) json_response(['success' => false, 'error' => 'Question not found'], 404);

$courseId = (string)($row['courseId'] ?? '');

$options = json_decode((string)$row['options'], true) ?: [];
$answerIndex = (int)$row['answerIndex'];
$isCorrect = ($selectedIndex === $answerIndex);

try {
  $pdo->beginTransaction();

  // ---- log attempt (for analytics, trends, timed mode)
  $pdo->prepare('
    INSERT INTO user_question_attempts (user_id, bank_id, question_id, selected_index, is_correct, seconds_spent, created_at)
    VALUES (:uid, :bid, :qid, :sel, :isc, :sec, UTC_TIMESTAMP())
  ')->execute([
    'uid' => $me['id'],
    'bid' => $bankId,
    'qid' => $qid,
    'sel' => $selectedIndex,
    'isc' => $isCorrect ? 1 : 0,
    'sec' => max(0, $secondsSpent),
  ]);

  // ---- spaced-repetition "wrong questions" queue (uses user_wrong_questions table)
  // Only creates/updates a record when the user gets it wrong OR already has it in the queue.
  $uwqStmt = $pdo->prepare('
    SELECT status, repetitions, interval_days, ease, wrong_count
         , next_review_at
    FROM user_wrong_questions
    WHERE user_id = :uid AND bank_id = :bid AND question_id = :qid
    LIMIT 1
  ');
  $uwqStmt->execute(['uid' => $me['id'], 'bid' => $bankId, 'qid' => $qid]);
  $uwq = $uwqStmt->fetch();

  $now = new DateTimeImmutable('now', new DateTimeZone('UTC'));
  $wasDueReview = false;
  if ($uwq) {
    try {
      $st = (string)($uwq['status'] ?? 'active');
      $nra = (string)($uwq['next_review_at'] ?? '');
      if ($st === 'active' && $nra !== '' && strtotime($nra) !== false && strtotime($nra) <= $now->getTimestamp()) {
        $wasDueReview = true;
      }
    } catch (Throwable $e) {
      $wasDueReview = false;
    }

    $status = (string)($uwq['status'] ?? 'active');
    $reps = (int)($uwq['repetitions'] ?? 0);
    $interval = (int)($uwq['interval_days'] ?? 1);
    $ease = (float)($uwq['ease'] ?? 2.30);
    $wrongCount = (int)($uwq['wrong_count'] ?? 0);

    if ($isCorrect) {
      $reps = $reps + 1;
      // SM-2-ish scheduling (simple + practical)
      if ($reps === 1) $interval = 1;
      else if ($reps === 2) $interval = 3;
      else $interval = max(1, (int)round($interval * $ease));

      $ease = min(2.50, $ease + 0.05);
      if ($reps >= 5) $status = 'mastered';
    } else {
      $wrongCount += 1;
      $reps = 0;
      $interval = 1;
      $ease = max(1.30, $ease - 0.20);
      $status = 'active';
    }

    $next = $now->modify('+' . $interval . ' days')->format('Y-m-d H:i:s');
    $pdo->prepare('
      UPDATE user_wrong_questions
      SET course_id = :cid,
          status = :st,
          repetitions = :r,
          interval_days = :i,
          ease = :e,
          wrong_count = :w,
          last_review_at = UTC_TIMESTAMP(),
          next_review_at = :n,
          updated_at = UTC_TIMESTAMP()
      WHERE user_id = :uid AND bank_id = :bid AND question_id = :qid
    ')->execute([
      'cid' => $courseId,
      'st'  => $status,
      'r'   => $reps,
      'i'   => $interval,
      'e'   => number_format($ease, 2, '.', ''),
      'w'   => $wrongCount,
      'n'   => $next,
      'uid' => $me['id'],
      'bid' => $bankId,
      'qid' => $qid,
    ]);
  } else {
    if (!$isCorrect) {
      $next = $now->modify('+1 days')->format('Y-m-d H:i:s');
      $pdo->prepare('
        INSERT INTO user_wrong_questions
          (user_id, course_id, bank_id, question_id, status, repetitions, interval_days, ease, wrong_count, last_review_at, next_review_at, created_at, updated_at)
        VALUES
          (:uid, :cid, :bid, :qid, "active", 0, 1, 2.30, 1, UTC_TIMESTAMP(), :n, UTC_TIMESTAMP(), UTC_TIMESTAMP())
      ')->execute([
        'uid' => $me['id'],
        'cid' => $courseId,
        'bid' => $bankId,
        'qid' => $qid,
        'n'   => $next,
      ]);
    }
  }


  // ---- user_bank_stats upsert
  $stmt = $pdo->prepare('
    SELECT answered_ids, correct_ids
    FROM user_bank_stats
    WHERE user_id = :uid AND bank_id = :bid
    LIMIT 1
  ');
  $stmt->execute(['uid' => $me['id'], 'bid' => $bankId]);
  $s = $stmt->fetch();

  $answered = $s ? (json_decode((string)$s['answered_ids'], true) ?: []) : [];
  $correct  = $s ? (json_decode((string)$s['correct_ids'], true) ?: []) : [];

  if (!in_array($qid, $answered, true)) $answered[] = $qid;
  if ($isCorrect && !in_array($qid, $correct, true)) $correct[] = $qid;

  if ($s) {
    $pdo->prepare('
      UPDATE user_bank_stats
      SET answered_ids = :a,
          correct_ids = :c,
          updated_at = UTC_TIMESTAMP()
      WHERE user_id = :uid AND bank_id = :bid
    ')->execute([
      'a'   => json_encode(array_values($answered), JSON_UNESCAPED_UNICODE),
      'c'   => json_encode(array_values($correct), JSON_UNESCAPED_UNICODE),
      'uid' => $me['id'],
      'bid' => $bankId,
    ]);
  } else {
    $pdo->prepare('
      INSERT INTO user_bank_stats (user_id, bank_id, answered_ids, correct_ids, updated_at)
      VALUES (:uid, :bid, :a, :c, UTC_TIMESTAMP())
    ')->execute([
      'uid' => $me['id'],
      'bid' => $bankId,
      'a'   => json_encode(array_values($answered), JSON_UNESCAPED_UNICODE),
      'c'   => json_encode(array_values($correct), JSON_UNESCAPED_UNICODE),
    ]);
  }

  // ---- user_progress upsert
  $pr = null;
  $streakFreezes = 0;
  try {
    $p = $pdo->prepare('SELECT streak, total_answered, correct_answered, study_seconds, last_active, streak_freezes
                        FROM user_progress WHERE user_id = :uid LIMIT 1');
    $p->execute(['uid' => $me['id']]);
    $pr = $p->fetch();
    $streakFreezes = (int)($pr['streak_freezes'] ?? 0);
  } catch (Throwable $e) {
    // migration not applied yet
    $p = $pdo->prepare('SELECT streak, total_answered, correct_answered, study_seconds, last_active
                        FROM user_progress WHERE user_id = :uid LIMIT 1');
    $p->execute(['uid' => $me['id']]);
    $pr = $p->fetch();
    $streakFreezes = 0;
  }

  $today = today_iso_date();
  $yesterday = (new DateTimeImmutable($today, new DateTimeZone('UTC')))
    ->modify('-1 day')
    ->format('Y-m-d');

  $streak = 1;
  $totalAnswered = 0;
  $correctAnswered = 0;
  $studySeconds = 0;
  $lastActive = $today;

  if ($pr) {
    $streak = (int)$pr['streak'];
    $totalAnswered = (int)$pr['total_answered'];
    $correctAnswered = (int)$pr['correct_answered'];
    $studySeconds = (int)$pr['study_seconds'];
    $lastActive = (string)$pr['last_active'];
  } else {
    try {
      $pdo->prepare('INSERT INTO user_progress (user_id, streak, accuracy, total_answered, correct_answered, study_seconds, last_active, streak_freezes)
                     VALUES (:uid, 1, 0, 0, 0, 0, :d, 0)')
        ->execute(['uid' => $me['id'], 'd' => $today]);
    } catch (Throwable $e) {
      $pdo->prepare('INSERT INTO user_progress (user_id, streak, accuracy, total_answered, correct_answered, study_seconds, last_active)
                     VALUES (:uid, 1, 0, 0, 0, 0, :d)')
        ->execute(['uid' => $me['id'], 'd' => $today]);
    }
  }

  if ($lastActive !== $today) {
    if ($lastActive === $yesterday) {
      $streak = max(1, $streak + 1);
    } else {
      // Streak freeze: if you missed exactly 1 day (gap=2) and have a freeze, keep streak.
      try {
        $dLast = new DateTimeImmutable($lastActive, new DateTimeZone('UTC'));
        $dToday = new DateTimeImmutable($today, new DateTimeZone('UTC'));
        $gap = (int)$dLast->diff($dToday)->format('%a');
        if ($gap === 2 && $streakFreezes > 0) {
          $streakFreezes = max(0, $streakFreezes - 1);
          $streak = max(1, $streak + 1);
        } else {
          $streak = 1;
        }
      } catch (Throwable $e) {
        $streak = 1;
      }
    }
  }

  $totalAnswered += 1;
  if ($isCorrect) $correctAnswered += 1;
  $studySeconds += max(0, $secondsSpent);

  $accuracy = $totalAnswered ? (int)round(($correctAnswered / $totalAnswered) * 100) : 0;

  try {
    $pdo->prepare('UPDATE user_progress
      SET streak = :s,
          accuracy = :a,
          total_answered = :t,
          correct_answered = :c,
          study_seconds = :ss,
          last_active = :d,
          streak_freezes = :sf
      WHERE user_id = :uid'
    )->execute([
      's' => $streak,
      'a' => $accuracy,
      't' => $totalAnswered,
      'c' => $correctAnswered,
      'ss' => $studySeconds,
      'd' => $today,
      'sf' => $streakFreezes,
      'uid' => $me['id'],
    ]);
  } catch (Throwable $e) {
    $pdo->prepare('UPDATE user_progress
      SET streak = :s,
          accuracy = :a,
          total_answered = :t,
          correct_answered = :c,
          study_seconds = :ss,
          last_active = :d
      WHERE user_id = :uid'
    )->execute([
      's' => $streak,
      'a' => $accuracy,
      't' => $totalAnswered,
      'c' => $correctAnswered,
      'ss' => $studySeconds,
      'd' => $today,
      'uid' => $me['id'],
    ]);
  }

  // ---- Weekly Missions (best-effort)
  $missionsClaimable = null;
  try {
    $missionsHelper = __DIR__ . '/../gamification/_missions.php';
    if (is_file($missionsHelper)) {
      require_once $missionsHelper;
      $weekStart = missions_week_start_utc();
      missions_increment($pdo, (string)$me['id'], [
        'answered' => 1,
        'correct' => $isCorrect ? 1 : 0,
        'studySeconds' => max(0, $secondsSpent),
        'dueReviews' => ($wasDueReview && $isCorrect) ? 1 : 0,
      ], $weekStart);
      $missionsClaimable = missions_claimable_count($pdo, (string)$me['id'], $weekStart);
    }
  } catch (Throwable $e) {
    $missionsClaimable = null;
  }

  /**
   * Gamification (optional): XP, levels, badges, daily goal progress.
   * This should never break practice if the table isn't migrated yet.
   */
  $gOut = null;
  try {
    // Pull current gamification row (or create one)
    $g = $pdo->prepare('SELECT daily_goal, xp, level, badges_json FROM user_gamification WHERE user_id = :uid LIMIT 1');
    $g->execute([':uid' => $me['id']]);
    $gr = $g->fetch();

    $dailyGoal = 10;
    $xp = 0;
    $level = 1;
    $badges = [];

    if ($gr) {
      $dailyGoal = max(5, min(200, (int)($gr['daily_goal'] ?? 10)));
      $xp = (int)($gr['xp'] ?? 0);
      $level = max(1, (int)($gr['level'] ?? 1));
      $badges = json_decode((string)($gr['badges_json'] ?? '[]'), true) ?: [];
      if (!is_array($badges)) $badges = [];
    } else {
      // Insert a baseline row
      $pdo->prepare('INSERT INTO user_gamification (user_id, daily_goal, xp, level, badges_json, updated_at)
                     VALUES (:uid, 10, 0, 1, :b, UTC_TIMESTAMP())')
        ->execute([
          'uid' => $me['id'],
          'b' => json_encode([], JSON_UNESCAPED_UNICODE),
        ]);
      $dailyGoal = 10;
      $xp = 0;
      $level = 1;
      $badges = [];
    }

    // Today counts (includes the attempt we just inserted)
    $t = $pdo->prepare('
      SELECT
        COUNT(*) AS answered,
        SUM(CASE WHEN is_correct = 1 THEN 1 ELSE 0 END) AS correct
      FROM user_question_attempts
      WHERE user_id = :uid AND DATE(created_at) = UTC_DATE()
    ');
    $t->execute(['uid' => $me['id']]);
    $tr = $t->fetch() ?: [];
    $todayAnswered = (int)($tr['answered'] ?? 0);
    $todayCorrect = (int)($tr['correct'] ?? 0);

    // XP gain: small + consistent (keeps students practicing)
    $xpEarned = 8 + ($isCorrect ? 5 : 0);
    $xpNew = max(0, $xp + $xpEarned);

    // Level curve: 250 XP per level (simple + predictable)
    $levelNew = (int)floor($xpNew / 250) + 1;
    $levelUp = $levelNew > $level;

    // Badge logic (keys)
    $wants = [];
    if ($totalAnswered >= 10) $wants[] = 'first_10';
    if ($totalAnswered >= 100) $wants[] = 'first_100';
    if ($totalAnswered >= 500) $wants[] = 'first_500';
    if ($streak >= 7) $wants[] = 'streak_7';
    if ($streak >= 30) $wants[] = 'streak_30';
    if ($accuracy >= 80 && $totalAnswered >= 50) $wants[] = 'accuracy_80';
    if ($todayAnswered >= $dailyGoal && $dailyGoal >= 10) $wants[] = 'goal_complete';
    if ($todayAnswered >= 10 && $todayCorrect === $todayAnswered) $wants[] = 'perfect_10';

    $badgeSet = array_values(array_unique(array_filter(array_map('strval', $badges))));
    $newBadges = [];
    foreach ($wants as $k) {
      if (!in_array($k, $badgeSet, true)) {
        $badgeSet[] = $k;
        $newBadges[] = $k;
      }
    }

    $pdo->prepare('UPDATE user_gamification
      SET daily_goal = :g,
          xp = :xp,
          level = :lvl,
          badges_json = :b,
          updated_at = UTC_TIMESTAMP()
      WHERE user_id = :uid'
    )->execute([
      'g' => $dailyGoal,
      'xp' => $xpNew,
      'lvl' => $levelNew,
      'b' => json_encode(array_values($badgeSet), JSON_UNESCAPED_UNICODE),
      'uid' => $me['id'],
    ]);

    $gOut = [
      'dailyGoal' => $dailyGoal,
      'todayAnswered' => $todayAnswered,
      'todayCorrect' => $todayCorrect,
      'xp' => $xpNew,
      'level' => $levelNew,
      'xpEarned' => $xpEarned,
      'levelUp' => $levelUp,
      'badges' => array_values($badgeSet),
      'badgesNew' => $newBadges,
    ];
  } catch (Throwable $e) {
    $gOut = null;
  }

  $pdo->commit();

  json_response(['success' => true, 'data' => [
    'result' => [
      'bankId'        => $bankId,
      'questionId'    => $qid,
      'selectedIndex' => $selectedIndex,
      'answerIndex'   => $answerIndex,
      'isCorrect'     => $isCorrect,
      'explain'       => $row['explanationText'] ?? '',
    ],
    'progress' => array_merge(
      [
        'streak'          => $streak,
        'accuracy'        => $accuracy,
        'totalAnswered'   => $totalAnswered,
        'correctAnswered' => $correctAnswered,
        'studySeconds'    => $studySeconds,
        'lastActive'      => $today,
        'streakFreezes'   => $streakFreezes,
        'missionsClaimable' => $missionsClaimable,

        // ✅ Resume helpers (for Dashboard "Continue where you left off")
        'lastBankId'      => $bankId,
        'lastQuestionId'  => $qid,
        'lastAttemptAt'   => gmdate('Y-m-d H:i:s'),
      ],
      is_array($gOut) ? $gOut : []
    ),
    'bankStats' => [
      'answeredIds' => $answered,
      'correctIds'  => $correct,
    ],
  ]]);
} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  json_response(['success' => false, 'error' => 'Failed to submit answer'], 500);
}
