<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../auth.php';
// Notify helpers (optional)
$notify_helpers = __DIR__ . '/../notify/_helpers.php';
if (is_file($notify_helpers)) {
  require_once $notify_helpers;
}

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'GET') {
  json_response([
    'success' => false,
    'error' => 'Method not allowed'
  ], 405);
}

$me = require_auth($pdo);

/**
 * Progress
 */
$pRow = null;
try {
  $stmt = $pdo->prepare(
    'SELECT streak,
            accuracy,
            total_answered AS totalAnswered,
            correct_answered AS correctAnswered,
            study_seconds AS studySeconds,
            last_active AS lastActive,
            streak_freezes AS streakFreezes
     FROM user_progress
     WHERE user_id = :uid
     LIMIT 1'
  );
  $stmt->execute([':uid' => $me['id']]);
  $pRow = $stmt->fetch();
} catch (Throwable $e) {
  // migration not applied yet
  $stmt = $pdo->prepare(
    'SELECT streak,
            accuracy,
            total_answered AS totalAnswered,
            correct_answered AS correctAnswered,
            study_seconds AS studySeconds,
            last_active AS lastActive
     FROM user_progress
     WHERE user_id = :uid
     LIMIT 1'
  );
  $stmt->execute([':uid' => $me['id']]);
  $pRow = $stmt->fetch();
}

$pRow = $pRow ?: [
  'streak' => 1,
  'accuracy' => 0,
  'totalAnswered' => 0,
  'correctAnswered' => 0,
  'studySeconds' => 0,
  'lastActive' => gmdate('Y-m-d'),
  'streakFreezes' => 0,
];

$p = [
  'streak' => (int)$pRow['streak'],
  'accuracy' => (int)$pRow['accuracy'],
  'totalAnswered' => (int)$pRow['totalAnswered'],
  'correctAnswered' => (int)$pRow['correctAnswered'],
  'studySeconds' => (int)$pRow['studySeconds'],
  'lastActive' => (string)$pRow['lastActive'],
  'streakFreezes' => (int)($pRow['streakFreezes'] ?? 0),
];

/**
 * Gamification (optional table)
 */
$gamification = [
  'dailyGoal' => 10,
  'xp' => 0,
  'level' => 1,
  'badges' => [],
];

try {
  $g = $pdo->prepare('SELECT daily_goal, xp, level, badges_json FROM user_gamification WHERE user_id = :uid LIMIT 1');
  $g->execute([':uid' => $me['id']]);
  $gr = $g->fetch();
  if ($gr) {
    $gamification['dailyGoal'] = max(5, min(200, (int)($gr['daily_goal'] ?? 10)));
    $gamification['xp'] = (int)($gr['xp'] ?? 0);
    $gamification['level'] = max(1, (int)($gr['level'] ?? 1));
    $gamification['badges'] = json_decode((string)($gr['badges_json'] ?? '[]'), true) ?: [];
  }
} catch (Throwable $e) {
  // table not migrated yet
}

/**
 * Today stats (computed from attempts)
 */
$todayAnswered = 0;
$todayCorrect = 0;
try {
  $t = $pdo->prepare('
    SELECT
      COUNT(*) AS answered,
      SUM(CASE WHEN is_correct = 1 THEN 1 ELSE 0 END) AS correct
    FROM user_question_attempts
    WHERE user_id = :uid AND DATE(created_at) = UTC_DATE()
  ');
  $t->execute(['uid' => $me['id']]);
  $tr = $t->fetch() ?: [];
  $todayAnswered = (int)($tr['answered'] ?? 0);
  $todayCorrect = (int)($tr['correct'] ?? 0);
} catch (Throwable $e) {
  $todayAnswered = 0;
  $todayCorrect = 0;
}

/**
 * Saved items
 */
$saved = [
  'pastQuestions' => [],
  'materials' => [],
  'questions' => [],
];

$s = $pdo->prepare(
  'SELECT kind, item_id
   FROM saved_items
   WHERE user_id = :uid
   ORDER BY created_at DESC'
);
$s->execute([':uid' => $me['id']]);

foreach ($s->fetchAll() ?: [] as $r) {
  $kind = (string)$r['kind'];
  if (array_key_exists($kind, $saved)) {
    $saved[$kind][] = (string)$r['item_id'];
  }
}

/**
 * Bank stats
 */
$answers = [];
$b = $pdo->prepare(
  'SELECT bank_id, answered_ids, correct_ids
   FROM user_bank_stats
   WHERE user_id = :uid'
);
$b->execute([':uid' => $me['id']]);

foreach ($b->fetchAll() ?: [] as $r) {
  $answers[(string)$r['bank_id']] = [
    'answeredIds' => json_decode((string)$r['answered_ids'], true) ?: [],
    'correctIds' => json_decode((string)$r['correct_ids'], true) ?: [],
  ];
}





/**
 * Theory bank stats (best effort)
 * 
 * Returned as: theoryAnswers[bankId] = { attemptedCount, avgScore, lastAttemptAt }
 */
$theoryAnswers = [];
$todayTheoryAnswered = 0;
try {
  $st = $pdo->prepare('
    SELECT bank_id,
           COUNT(DISTINCT question_id) AS attemptedCount,
           AVG(self_score) AS avgScore,
           MAX(created_at) AS lastAttemptAt
    FROM user_theory_attempts
    WHERE user_id = :uid
    GROUP BY bank_id
  ');
  $st->execute([':uid' => $me['id']]);
  foreach ($st->fetchAll() ?: [] as $r) {
    $bid = (string)($r['bank_id'] ?? '');
    if ($bid === '') continue;
    $avg = $r['avgScore'];
    $theoryAnswers[$bid] = [
      'attemptedCount' => (int)($r['attemptedCount'] ?? 0),
      'avgScore' => ($avg === null ? null : (float)$avg),
      'lastAttemptAt' => $r['lastAttemptAt'] === null ? null : (string)$r['lastAttemptAt'],
    ];
  }

  $t2 = $pdo->prepare('SELECT COUNT(*) AS c FROM user_theory_attempts WHERE user_id = :uid AND DATE(created_at) = UTC_DATE()');
  $t2->execute([':uid' => $me['id']]);
  $todayTheoryAnswered = (int)(($t2->fetch() ?: [])['c'] ?? 0);
} catch (Throwable $e) {
  $theoryAnswers = [];
  $todayTheoryAnswered = 0;
}

$due = 0;
try {
  $cnt = $pdo->prepare('
    SELECT COUNT(*) AS c
    FROM user_wrong_questions
    WHERE user_id = :uid AND status = "active" AND next_review_at <= UTC_TIMESTAMP()
  ');
  $cnt->execute(['uid' => $me['id']]);
  $due = (int)(($cnt->fetch() ?: [])['c'] ?? 0);
} catch (Throwable $e) {
  $due = 0;
}


$notifyUnread = 0;
try {
  if (function_exists('notify_unread_count')) {
    $notifyUnread = notify_unread_count($pdo, (string)$me['id']);
  }
} catch (Throwable $e) {
  $notifyUnread = 0;
}

/**
 * Resume / last activity (best-effort)
 * Used for "Continue where you left off" deep-links.
 */
$lastBankId = null;
$lastQuestionId = null;
$lastAttemptAt = null;
try {
  $la = $pdo->prepare(
    'SELECT bank_id AS bankId, question_id AS questionId, created_at AS createdAt
     FROM user_question_attempts
     WHERE user_id = :uid
     ORDER BY created_at DESC, id DESC
     LIMIT 1'
  );
  $la->execute(['uid' => $me['id']]);
  $lr = $la->fetch() ?: null;
  if ($lr) {
    $lastBankId = (string)($lr['bankId'] ?? '');
    $lastQuestionId = (string)($lr['questionId'] ?? '');
    $lastAttemptAt = (string)($lr['createdAt'] ?? '');
  }
} catch (Throwable $e) {
  // table may not exist yet
}


json_response([
  'success' => true,
  'data' => [
    'progress' => array_merge(
      $p,
      [
        'saved' => $saved,
        'dueReviews' => $due,
        // Gamification
        'dailyGoal' => (int)($gamification['dailyGoal'] ?? 10),
        'xp' => (int)($gamification['xp'] ?? 0),
        'level' => (int)($gamification['level'] ?? 1),
        'badges' => is_array($gamification['badges'] ?? null) ? $gamification['badges'] : [],
        // Today
        'todayAnswered' => $todayAnswered,
        'todayCorrect' => $todayCorrect,
        'todayTheoryAnswered' => $todayTheoryAnswered,
        // Notify
        'notifyUnread' => (int)$notifyUnread,

        // Resume
        'lastBankId' => $lastBankId,
        'lastQuestionId' => $lastQuestionId,
        'lastAttemptAt' => $lastAttemptAt,
      ]
    ),
    'answers' => $answers,
    'theoryAnswers' => $theoryAnswers,
  ]
]);
