<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);

$body = read_json();
require_fields($body, ['courseIds', 'level', 'reason']);

$courseIds = $body['courseIds'];
$level = (int)$body['level'];
$reason = trim((string)$body['reason']);
$proofUrl = trim((string)($body['proofUrl'] ?? ''));

if (!is_array($courseIds) || count($courseIds) < 1) {
  json_response(['success' => false, 'error' => 'courseIds must be a non-empty array'], 422);
}
$courseIds = array_values(array_unique(array_filter(array_map(fn($c) => trim((string)$c), $courseIds))));
if (count($courseIds) < 1) {
  json_response(['success' => false, 'error' => 'courseIds must contain valid course IDs'], 422);
}
if ($level < 100 || $level > 700) {
  json_response(['success' => false, 'error' => 'level looks invalid'], 422);
}
if ($reason === '') {
  json_response(['success' => false, 'error' => 'reason is required'], 422);
}

// Only one pending request at a time per user
// NOTE: don't embed escaped quotes (\") in SQL; bind values instead.
$stmt = $pdo->prepare('SELECT id FROM course_rep_requests WHERE user_id = :uid AND status = :status LIMIT 1');
$stmt->execute([':uid' => $me['id'], ':status' => 'pending']);
$pending = $stmt->fetchColumn();
if ($pending) {
  json_response(['success' => false, 'error' => 'You already have a pending request'], 409);
}

$rid = uuidv4();

try {
  $pdo->beginTransaction();

  $pdo->prepare('INSERT INTO course_rep_requests (id, user_id, level, reason, proof_url, status, requested_at)
                 VALUES (:id,:uid,:lvl,:rea,:proof,:status,UTC_TIMESTAMP())')
      ->execute([
        ':id' => $rid,
        ':uid' => $me['id'],
        ':lvl' => $level,
        ':rea' => $reason,
        ':proof' => $proofUrl !== '' ? $proofUrl : null,
        ':status' => 'pending',
      ]);

  $ins = $pdo->prepare('INSERT INTO course_rep_request_courses (request_id, course_id) VALUES (:rid,:cid)');
  foreach ($courseIds as $cid) {
    $ins->execute([':rid' => $rid, ':cid' => $cid]);
  }

  $pdo->commit();

  json_response(['success' => true, 'data' => [
    'request' => [
      'id' => $rid,
      'status' => 'pending',
      'level' => $level,
      'reason' => $reason,
      'proofUrl' => $proofUrl,
      'courseIds' => $courseIds,
      'requestedAt' => now_iso(),
    ]
  ]], 201);
} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  json_response(['success' => false, 'error' => 'Failed to submit request'], 500);
}
