<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../auth.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'GET') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);

$bankId = trim((string)($_GET['bankId'] ?? ''));
$questionId = trim((string)($_GET['questionId'] ?? ''));
$limit = (int)($_GET['limit'] ?? 80);
if ($limit <= 0) $limit = 80;
if ($limit > 300) $limit = 300;

if ($bankId === '') {
  json_response(['success' => false, 'error' => 'Missing bankId'], 422);
}

try {
  $where = 'user_id = :uid AND bank_id = :bid';
  $params = [':uid' => (string)$me['id'], ':bid' => $bankId];
  if ($questionId !== '') {
    $where .= ' AND question_id = :qid';
    $params[':qid'] = $questionId;
  }

  // Newer schema may include AI feedback columns. Query them if present,
  // but degrade gracefully if the migration hasn't been run yet.
  try {
    // Preferred column name (some deployments)
    $sql = 'SELECT id,
                   question_id AS questionId,
                   answer_text AS answerText,
                   self_score AS selfScore,
                   ai_score AS aiScore,
                   ai_feedback_json AS aiFeedbackJson,
                   ai_model AS aiModel,
                   ai_updated_at AS aiUpdatedAt,
                   seconds_spent AS secondsSpent,
                   created_at AS createdAt
            FROM user_theory_attempts
            WHERE ' . $where . '
            ORDER BY created_at DESC, id DESC
            LIMIT ' . $limit;

    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $rows = $stmt->fetchAll() ?: [];
  } catch (Throwable $e) {
    try {
      // Alternate column name (migration 2026_01_11_ai_theory_coach.sql)
      $sql = 'SELECT id,
                     question_id AS questionId,
                     answer_text AS answerText,
                     self_score AS selfScore,
                     ai_score AS aiScore,
                     ai_feedback_json AS aiFeedbackJson,
                     ai_model AS aiModel,
                     ai_created_at AS aiUpdatedAt,
                     seconds_spent AS secondsSpent,
                     created_at AS createdAt
              FROM user_theory_attempts
              WHERE ' . $where . '
              ORDER BY created_at DESC, id DESC
              LIMIT ' . $limit;

      $stmt = $pdo->prepare($sql);
      $stmt->execute($params);
      $rows = $stmt->fetchAll() ?: [];
    } catch (Throwable $e2) {
      // Fallback (older schema)
      $sql = 'SELECT id,
                     question_id AS questionId,
                     answer_text AS answerText,
                     self_score AS selfScore,
                     seconds_spent AS secondsSpent,
                     created_at AS createdAt
              FROM user_theory_attempts
              WHERE ' . $where . '
              ORDER BY created_at DESC, id DESC
              LIMIT ' . $limit;

      $stmt = $pdo->prepare($sql);
      $stmt->execute($params);
      $rows = $stmt->fetchAll() ?: [];
    }
  }

  // Latest attempt per question (for quick UI marking)
  $latestByQuestion = [];
  foreach ($rows as $r) {
    $qid = (string)($r['questionId'] ?? '');
    if ($qid === '') continue;
    if (!isset($latestByQuestion[$qid])) {
        $aiFeedback = null;
        if (isset($r['aiFeedbackJson']) && is_string($r['aiFeedbackJson']) && trim((string)$r['aiFeedbackJson']) !== '') {
          $tmp = json_decode((string)$r['aiFeedbackJson'], true);
          if (is_array($tmp)) $aiFeedback = $tmp;
        }

        $latestByQuestion[$qid] = [
        'attemptId' => (string)($r['id'] ?? ''),
        'questionId' => $qid,
        'answerText' => (string)($r['answerText'] ?? ''),
        'selfScore' => ($r['selfScore'] === null ? null : (int)$r['selfScore']),
          'aiScore' => (isset($r['aiScore']) && $r['aiScore'] !== null ? (int)$r['aiScore'] : null),
          'aiFeedback' => $aiFeedback,
          'aiModel' => isset($r['aiModel']) ? (string)$r['aiModel'] : null,
          'aiUpdatedAt' => isset($r['aiUpdatedAt']) ? (string)$r['aiUpdatedAt'] : null,
        'secondsSpent' => (int)($r['secondsSpent'] ?? 0),
        'createdAt' => (string)($r['createdAt'] ?? ''),
      ];
    }
  }

  // Bank stats (attemptedCount/avgScore)
  $bankStats = null;
  try {
    $bs = $pdo->prepare('SELECT COUNT(DISTINCT question_id) AS attemptedCount,
                                AVG(self_score) AS avgScore,
                                MAX(created_at) AS lastAttemptAt
                         FROM user_theory_attempts
                         WHERE user_id = :uid AND bank_id = :bid');
    $bs->execute([':uid' => (string)$me['id'], ':bid' => $bankId]);
    $r = $bs->fetch() ?: null;
    if ($r) {
      $bankStats = [
        'attemptedCount' => (int)($r['attemptedCount'] ?? 0),
        'avgScore' => ($r['avgScore'] === null ? null : (float)$r['avgScore']),
        'lastAttemptAt' => $r['lastAttemptAt'] === null ? null : (string)$r['lastAttemptAt'],
      ];
    }
  } catch (Throwable $e) {
    $bankStats = null;
  }

  json_response([
    'success' => true,
    'data' => [
      'attempts' => array_map(static function ($r) {
        $aiFeedback = null;
        if (isset($r['aiFeedbackJson']) && is_string($r['aiFeedbackJson']) && trim((string)$r['aiFeedbackJson']) !== '') {
          $tmp = json_decode((string)$r['aiFeedbackJson'], true);
          if (is_array($tmp)) $aiFeedback = $tmp;
        }
        return [
          'attemptId' => (string)($r['id'] ?? ''),
          'questionId' => (string)($r['questionId'] ?? ''),
          'answerText' => (string)($r['answerText'] ?? ''),
          'selfScore' => ($r['selfScore'] === null ? null : (int)$r['selfScore']),
          'aiScore' => (isset($r['aiScore']) && $r['aiScore'] !== null ? (int)$r['aiScore'] : null),
          'aiFeedback' => $aiFeedback,
          'aiModel' => isset($r['aiModel']) ? (string)$r['aiModel'] : null,
          'aiUpdatedAt' => isset($r['aiUpdatedAt']) ? (string)$r['aiUpdatedAt'] : null,
          'secondsSpent' => (int)($r['secondsSpent'] ?? 0),
          'createdAt' => (string)($r['createdAt'] ?? ''),
        ];
      }, $rows),
      'latestByQuestion' => $latestByQuestion,
      'bankStats' => $bankStats,
    ],
  ]);

} catch (Throwable $e) {
  json_response([
    'success' => false,
    'error' => 'Failed to fetch theory attempts',
    'detail' => $e->getMessage(),
  ], 500);
}
