<?php
declare(strict_types=1);

require_once __DIR__ . '/../response.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/../auth.php';

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['success' => false, 'error' => 'Method not allowed'], 405);
}

$me = require_auth($pdo);
$body = read_json();
require_fields($body, ['bankId', 'questionId', 'answerText']);

$bankId = trim((string)$body['bankId']);
$questionId = trim((string)$body['questionId']);
$answerText = trim((string)$body['answerText']);
$selfScore = $body['selfScore'] ?? null;
$secondsSpent = (int)($body['secondsSpent'] ?? 0);

if ($bankId === '' || $questionId === '') {
  json_response(['success' => false, 'error' => 'Missing bankId/questionId'], 422);
}

if ($answerText === '') {
  json_response(['success' => false, 'error' => 'Answer cannot be empty'], 422);
}

if ($secondsSpent < 0) $secondsSpent = 0;
if ($secondsSpent > 60 * 60) $secondsSpent = 60 * 60; // clamp to 1 hour per question

$score = null;
if ($selfScore !== null && $selfScore !== '') {
  $score = (int)$selfScore;
  if ($score < 0) $score = 0;
  if ($score > 5) $score = 5; // 0..5 rubric
}

// Ensure bank exists + is theory
$stmt = $pdo->prepare('SELECT id, course_id AS courseId, title, mode FROM banks WHERE id = :id LIMIT 1');
$stmt->execute([':id' => $bankId]);
$bank = $stmt->fetch();
if (!$bank) {
  json_response(['success' => false, 'error' => 'Bank not found'], 404);
}

if (strtolower((string)($bank['mode'] ?? '')) !== 'theory') {
  json_response(['success' => false, 'error' => 'This bank is not a theory bank'], 422);
}

// Ensure question exists in this bank
$q = $pdo->prepare('SELECT id FROM theory_questions WHERE id = :qid AND bank_id = :bid LIMIT 1');
$q->execute([':qid' => $questionId, ':bid' => $bankId]);
if (!$q->fetch()) {
  json_response(['success' => false, 'error' => 'Question not found in this bank'], 404);
}

try {
  $pdo->beginTransaction();

  // Insert attempt
  $ins = $pdo->prepare('INSERT INTO user_theory_attempts (user_id, bank_id, question_id, answer_text, self_score, seconds_spent, created_at)
                        VALUES (:uid,:bid,:qid,:a,:sc,:ss,UTC_TIMESTAMP())');
  $ins->execute([
    ':uid' => (string)$me['id'],
    ':bid' => $bankId,
    ':qid' => $questionId,
    ':a' => $answerText,
    ':sc' => $score,
    ':ss' => $secondsSpent,
  ]);
  $attemptId = (string)$pdo->lastInsertId();

  // ---- user_progress upsert (streak + study seconds)
  $pr = null;
  $streakFreezes = 0;
  $hasFreezeCol = true;
  try {
    $p = $pdo->prepare('SELECT streak, accuracy, total_answered, correct_answered, study_seconds, last_active, streak_freezes
                        FROM user_progress WHERE user_id = :uid LIMIT 1');
    $p->execute(['uid' => $me['id']]);
    $pr = $p->fetch();
    $streakFreezes = (int)($pr['streak_freezes'] ?? 0);
  } catch (Throwable $e) {
    $hasFreezeCol = false;
    $p = $pdo->prepare('SELECT streak, accuracy, total_answered, correct_answered, study_seconds, last_active
                        FROM user_progress WHERE user_id = :uid LIMIT 1');
    $p->execute(['uid' => $me['id']]);
    $pr = $p->fetch();
    $streakFreezes = 0;
  }

  $today = today_iso_date();
  $yesterday = (new DateTimeImmutable($today, new DateTimeZone('UTC')))
    ->modify('-1 day')
    ->format('Y-m-d');

  $streak = 1;
  $accuracy = 0;
  $totalAnswered = 0;
  $correctAnswered = 0;
  $studySeconds = 0;
  $lastActive = $today;

  if ($pr) {
    $streak = (int)($pr['streak'] ?? 1);
    $accuracy = (int)($pr['accuracy'] ?? 0);
    $totalAnswered = (int)($pr['total_answered'] ?? 0);
    $correctAnswered = (int)($pr['correct_answered'] ?? 0);
    $studySeconds = (int)($pr['study_seconds'] ?? 0);
    $lastActive = (string)($pr['last_active'] ?? $today);
  } else {
    // Create baseline progress row
    try {
      $pdo->prepare('INSERT INTO user_progress (user_id, streak, accuracy, total_answered, correct_answered, study_seconds, last_active, streak_freezes)
                     VALUES (:uid, 1, 0, 0, 0, 0, :d, 0)')
        ->execute(['uid' => $me['id'], 'd' => $today]);
      $hasFreezeCol = true;
    } catch (Throwable $e) {
      $pdo->prepare('INSERT INTO user_progress (user_id, streak, accuracy, total_answered, correct_answered, study_seconds, last_active)
                     VALUES (:uid, 1, 0, 0, 0, 0, :d)')
        ->execute(['uid' => $me['id'], 'd' => $today]);
      $hasFreezeCol = false;
    }
  }

  // Update streak if needed
  if ($lastActive !== $today) {
    if ($lastActive === $yesterday) {
      $streak = max(1, $streak + 1);
    } else {
      // Streak freeze: if you missed exactly 1 day (gap=2) and have a freeze, keep streak.
      try {
        $dLast = new DateTimeImmutable($lastActive, new DateTimeZone('UTC'));
        $dToday = new DateTimeImmutable($today, new DateTimeZone('UTC'));
        $gap = (int)$dLast->diff($dToday)->format('%a');
        if ($gap === 2 && $streakFreezes > 0) {
          $streakFreezes = max(0, $streakFreezes - 1);
          $streak = max(1, $streak + 1);
        } else {
          $streak = 1;
        }
      } catch (Throwable $e) {
        $streak = 1;
      }
    }
  }

  $studySeconds += max(0, $secondsSpent);

  try {
    if ($hasFreezeCol) {
      $pdo->prepare('UPDATE user_progress
        SET streak = :s,
            accuracy = :a,
            total_answered = :t,
            correct_answered = :c,
            study_seconds = :ss,
            last_active = :d,
            streak_freezes = :sf
        WHERE user_id = :uid'
      )->execute([
        's' => $streak,
        'a' => $accuracy,
        't' => $totalAnswered,
        'c' => $correctAnswered,
        'ss' => $studySeconds,
        'd' => $today,
        'sf' => $streakFreezes,
        'uid' => $me['id'],
      ]);
    } else {
      $pdo->prepare('UPDATE user_progress
        SET streak = :s,
            accuracy = :a,
            total_answered = :t,
            correct_answered = :c,
            study_seconds = :ss,
            last_active = :d
        WHERE user_id = :uid'
      )->execute([
        's' => $streak,
        'a' => $accuracy,
        't' => $totalAnswered,
        'c' => $correctAnswered,
        'ss' => $studySeconds,
        'd' => $today,
        'uid' => $me['id'],
      ]);
    }
  } catch (Throwable $e) {
    // ignore
  }

  // ---- Weekly Missions (best-effort)
  $missionsClaimable = null;
  try {
    $missionsHelper = __DIR__ . '/../gamification/_missions.php';
    if (is_file($missionsHelper)) {
      require_once $missionsHelper;
      $weekStart = missions_week_start_utc();
      missions_increment($pdo, (string)$me['id'], [
        'answered' => 1, // theory attempts count towards weekly answered
        'correct' => 0,
        'studySeconds' => max(0, $secondsSpent),
        'dueReviews' => 0,
      ], $weekStart);
      $missionsClaimable = missions_claimable_count($pdo, (string)$me['id'], $weekStart);
    }
  } catch (Throwable $e) {
    $missionsClaimable = null;
  }

  // ---- Gamification (XP/levels) best-effort
  $gOut = null;
  try {
    $g = $pdo->prepare('SELECT daily_goal, xp, level, badges_json FROM user_gamification WHERE user_id = :uid LIMIT 1');
    $g->execute([':uid' => $me['id']]);
    $gr = $g->fetch();

    $dailyGoal = 10;
    $xp = 0;
    $level = 1;
    $badges = [];

    if ($gr) {
      $dailyGoal = max(5, min(200, (int)($gr['daily_goal'] ?? 10)));
      $xp = (int)($gr['xp'] ?? 0);
      $level = max(1, (int)($gr['level'] ?? 1));
      $badges = json_decode((string)($gr['badges_json'] ?? '[]'), true) ?: [];
      if (!is_array($badges)) $badges = [];
    } else {
      $pdo->prepare('INSERT INTO user_gamification (user_id, daily_goal, xp, level, badges_json, updated_at)
                     VALUES (:uid, 10, 0, 1, :b, UTC_TIMESTAMP())')
        ->execute([
          'uid' => $me['id'],
          'b' => json_encode([], JSON_UNESCAPED_UNICODE),
        ]);
      $dailyGoal = 10;
      $xp = 0;
      $level = 1;
      $badges = [];
    }

    // Today counts (objective MCQ only)
    $t = $pdo->prepare('SELECT COUNT(*) AS answered,
                               SUM(CASE WHEN is_correct = 1 THEN 1 ELSE 0 END) AS correct
                        FROM user_question_attempts
                        WHERE user_id = :uid AND DATE(created_at) = UTC_DATE()');
    $t->execute(['uid' => $me['id']]);
    $tr = $t->fetch() ?: [];
    $todayAnswered = (int)($tr['answered'] ?? 0);
    $todayCorrect = (int)($tr['correct'] ?? 0);

    // XP gain: reward writing + self-marking.
    $len = strlen($answerText);
    $lenBonus = min(4, (int)floor($len / 400));
    $scoreBonus = ($score === null) ? 0 : min(5, max(0, (int)$score));
    $xpEarned = 6 + $lenBonus + $scoreBonus;

    $xpNew = max(0, $xp + $xpEarned);
    $levelNew = (int)floor($xpNew / 250) + 1;
    $levelUp = $levelNew > $level;

    // Keep existing badge logic (based mostly on objective stats)
    $badgeSet = array_values(array_unique(array_filter(array_map('strval', $badges))));

    $pdo->prepare('UPDATE user_gamification
      SET daily_goal = :g,
          xp = :xp,
          level = :lvl,
          badges_json = :b,
          updated_at = UTC_TIMESTAMP()
      WHERE user_id = :uid'
    )->execute([
      'g' => $dailyGoal,
      'xp' => $xpNew,
      'lvl' => $levelNew,
      'b' => json_encode(array_values($badgeSet), JSON_UNESCAPED_UNICODE),
      'uid' => $me['id'],
    ]);

    $gOut = [
      'dailyGoal' => $dailyGoal,
      'todayAnswered' => $todayAnswered,
      'todayCorrect' => $todayCorrect,
      'xp' => $xpNew,
      'level' => $levelNew,
      'xpEarned' => $xpEarned,
      'levelUp' => $levelUp,
      'badges' => array_values($badgeSet),
      'badgesNew' => [],
    ];
  } catch (Throwable $e) {
    $gOut = null;
  }

  // Bank stats (for UI updates)
  $bankStats = null;
  try {
    $bs = $pdo->prepare('SELECT COUNT(DISTINCT question_id) AS attemptedCount,
                                AVG(self_score) AS avgScore,
                                MAX(created_at) AS lastAttemptAt
                         FROM user_theory_attempts
                         WHERE user_id = :uid AND bank_id = :bid');
    $bs->execute([':uid' => $me['id'], ':bid' => $bankId]);
    $r = $bs->fetch() ?: null;
    if ($r) {
      $bankStats = [
        'attemptedCount' => (int)($r['attemptedCount'] ?? 0),
        'avgScore' => ($r['avgScore'] === null ? null : (float)$r['avgScore']),
        'lastAttemptAt' => $r['lastAttemptAt'] === null ? null : (string)$r['lastAttemptAt'],
      ];
    }
  } catch (Throwable $e) {
    $bankStats = null;
  }

  $pdo->commit();

  json_response([
    'success' => true,
    'data' => [
      'result' => [
        'attemptId' => $attemptId,
        'bankId' => $bankId,
        'questionId' => $questionId,
        'selfScore' => $score,
        'secondsSpent' => $secondsSpent,
      ],
      'bankStats' => $bankStats,
      'progress' => array_merge(
        [
          'streak' => $streak,
          'studySeconds' => $studySeconds,
          'lastActive' => $today,
          'streakFreezes' => $streakFreezes,
          'missionsClaimable' => $missionsClaimable,
        ],
        $gOut ? [
          'dailyGoal' => $gOut['dailyGoal'],
          'todayAnswered' => $gOut['todayAnswered'],
          'todayCorrect' => $gOut['todayCorrect'],
          'xp' => $gOut['xp'],
          'level' => $gOut['level'],
          'levelUp' => $gOut['levelUp'],
          'badges' => $gOut['badges'],
          'badgesNew' => $gOut['badgesNew'],
        ] : []
      ),
      'rewards' => $gOut ? [
        'xp' => (int)($gOut['xpEarned'] ?? 0),
      ] : null,
    ],
  ]);

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  json_response([
    'success' => false,
    'error' => 'Failed to submit theory attempt',
    'detail' => $e->getMessage(),
  ], 500);
}
